package ru.yandex.calendar.frontend.web.cmd.run.ui.layer;

import java.util.Optional;

import javax.annotation.Nullable;

import lombok.val;
import org.jdom.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.frontend.caldav.impl.YandexCaldavUrls;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ValidatableXmlCommand;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.logic.beans.generated.IcsFeed;
import ru.yandex.calendar.logic.beans.generated.IcsFeedFields;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerFields;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.ics.feed.IcsFeedManager;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerUserWithRelations;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.notification.NotificationXmlizer;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.sharing.perm.PermXmlizer;
import ru.yandex.calendar.logic.svc.DbSvcRoutines;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.base.AuxColl;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Returns all layers owned (created / shared) by current user
 * @author ssytnik
 */
public class CmdGetLayers extends ValidatableXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdGetLayers.class);

    private static final String CMD_TAG = "get-layers";

    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private DbSvcRoutines dbSvcRoutines;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private PermXmlizer permXmlizer;
    @Autowired
    private EnvironmentType environmentType;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private IcsFeedManager icsFeedManager;

    private Option<PassportUid> uid2O; // who (logic thinks) user is (possibly null/ANYONE)
    private String layerIdsStr; // optional filter (default: all user's layers) / required for *A, *AP cases

    public CmdGetLayers(AuthInfo ai) {
        super(CMD_TAG, ai);
        this.uid2O = uidO;
        this.layerIdsStr = null;
    }

    public CmdGetLayers(@Nullable AuthInfo ai, String tzId, String layerIdsStr, String privateToken) {
        super(CMD_TAG, ai, tzId);
        setMustHaveAuth(ai != null);

        boolean layerIdsSet = StringUtils.isNotEmpty(layerIdsStr);
        boolean privateTokenSet = StringUtils.isNotEmpty(privateToken);
        if (layerIdsSet == privateTokenSet) {
            logger.error("Ctor(): !!! INVALID: layer ids set? " + layerIdsSet + "; private token set? " + privateTokenSet);
        }
        this.uid2O = uidO;
        this.layerIdsStr = layerIdsStr;
        if (privateTokenSet) { setPrivateToken(privateToken); }
    }

    @Override
    protected void obtainPrivateResource(String privateToken) {
        Layer l = layerRoutines.getByPrivateToken(privateToken);
        this.uid2O = Option.of(l.getCreatorUid()); // as if layer creator was looking over his own layer
        this.layerIdsStr = String.valueOf(l.getId());
    }

    @Override
    public void validate() {
        RequestValidator.validate(RequestValidator.LONG_ARRAY, "layerIds", layerIdsStr);
    }

    @Override
    protected void buildXmlResponseV(final XmlCmdContext ctx) {
        val source = ActionSource.WEB;
        Option<ListF<Long>> layerIdsO = StringUtils.isNotEmpty(layerIdsStr) ?
            Option.of(Cf.x(AuxColl.splitToLongArray(layerIdsStr))) :
            Option.<ListF<Long>>empty();

        ListF<LayerUserWithRelations> lwrs;

        if (layerIdsO.isPresent()) {
            lwrs = layerRoutines.getLayerUsersWithRelationsByLayerIds(layerIdsO.get())
                    .filter(LayerUserWithRelations::layerUserIsLayerCreator);
        } else if (uid2O.isPresent()) {
            lwrs = layerRoutines.getLayerUsersWithRelationsByUid(uid2O.get(), Option.empty());
        } else {
            throw new CommandRunException("CmdGetLayers: either uid2 or layerIds (or both) must be set");
        }

        if (!layerIdsO.isPresent()) {
            lwrs = lwrs.filterNot(Cf.list(PassportSid.AFISHA, PassportSid.TV)
                    .containsF().compose(lu -> lu.getLayer().getSid())); // CAL-6585
        }

        // Get additional user information for shared layers
        ListF<PassportUid> settingsUids = lwrs.map(LayerUserWithRelations::getLayerCreatorUid).stableUnique();

        if (uid2O.isPresent()) {
            settingsUids = settingsUids.plus(uid2O);
        }
        MapF<PassportUid, Settings> settingsMap = settingsRoutines.getSettingsCommonByUidBatch(settingsUids);

        if (lwrs.size() > 0) {
            Element layersElement = new Element("layers");

            if (uid2O.isPresent()) {
                Email email = settingsMap.getOrThrow(uid2O.get()).getEmail();
                String caldavUserUrl = YandexCaldavUrls.userUrl(email, environmentType);
                layersElement.setAttribute("caldav-user-url", caldavUserUrl);
            }

            MapF<Long, ListF<Notification>> notificationsByLayerUserId = notificationDbManager
                    .getNotificationsByLayerUserIds(lwrs.map(LayerUserWithRelations::getLayerUserId)).toMap();
            MapF<Long, IcsFeed> icsFeedByLayerId = Cf2.flatBy2(icsFeedManager
                    .findIcsFeedsByLayerIds(lwrs.map(LayerUserWithRelations::getLayerId))).toMap();

            Option<UserInfo> user2InfoO = obtainUserInfoO(uid2O);

            for (LayerUserWithRelations lwr : lwrs) {
                val layer = lwr.getLayer();
                val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                // TODO: 2009-07-10 ssytnik: maybe it would be more safe to do
                //       permMgr.check(...), then delete layer-user on failure
                // stepancheg: agreed
                user2InfoO.toOptional().ifPresentOrElse(
                    user -> authorizer.ensureCanPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.LIST, source),
                    () -> authorizer.ensureCanPerformLayerAction(layerPermInfo, LayerAction.LIST, source)
                );

                ListF<Notification> notifications = notificationsByLayerUserId.getTs(lwr.getLayerUser().getId());
                // Layer info
                Element eLayer = new Element("layer");
                layer.appendXmlTo(eLayer, tz, LayerFields.ID, LayerFields.CREATOR_UID, LayerFields.TYPE, LayerFields.SID);
                CalendarXmlizer.appendDtfElm(eLayer, "perm-all", "list", tz);
                // WAS (instead, we do it ALWAYS now): else { XmlUtils.appendElm(eLayer, "perm-all", Layer.PERM_ALL); }
                // For shared layers, show information about creator user
                Settings userSettings = settingsMap.getOrThrow(layer.getCreatorUid()); // creator uid is not null
                SettingsRoutines.appendCreatorUserLogin(eLayer, userSettings);

                CalendarXmlizer.appendElm(eLayer, "name", lwr.getEvaluatedLayerName());
                CalendarXmlizer.appendElm(eLayer, "s-name", layerRoutines.evalLayerSName(layer));
                CalendarXmlizer.appendElm(eLayer, "css-class", LayerRoutines.evalCssClass(layer, lwr.getLayerUser()));
                if (LayerRoutines.isService(layer)) {
                    CalendarXmlizer.appendElm(eLayer, "domain-name", LayerRoutines.evalServiceDomainName(layer));
                }
                CalendarXmlizer.appendElm(eLayer, "is-visible-in-ui", lwr.getLayerUser().getIsVisibleInUi());
                CalendarXmlizer.appendElm(eLayer, "is-notify-changes", lwr.getLayerUser().getIsNotifyChanges());
                CalendarXmlizer.appendElm(eLayer, "affects-availability", lwr.getLayerUser().getAffectsAvailability());
                NotificationXmlizer.appendElmForWeb(eLayer, notifications);
                // Evaluate permissions
                permXmlizer.appendLayerPermElms(eLayer, user2InfoO, layer.getId(), source,
                        LayerAction.CREATE_EVENT
                    // 2009-04-29 ssytnik: moved to CmdGetEvent (CAL-1100): LayerAction.DETACH_EVENT
                );

                if (uid2O.isPresent()) {
                    CalendarXmlizer.appendElm(eLayer, "caldav-collection-url", YandexCaldavUrls.calendarUrl(
                            settingsMap.getOrThrow(uid2O.get()).getYandexEmail(),
                            layer.getId(), environmentType)
                    );
                }
                if (icsFeedByLayerId.containsKeyTs(layer.getId())) {
                    Element eFeedInfo = new Element("feed-info");
                    CalendarXmlizer.appendElm(eFeedInfo, "id", icsFeedByLayerId.getTs(layer.getId()).getId());
                    icsFeedByLayerId.getTs(layer.getId()).appendXmlTo(
                            eFeedInfo, tz, IcsFeedFields.URL, IcsFeedFields.LAST_SUCCESS_TS);
                    eLayer.addContent(eFeedInfo);
                }

                layersElement.addContent(eLayer);
            }

            // If no layers found, do not append this block to output xml
            ctx.getRootElement().addContent(layersElement);
        }

        if (
            uidO.isPresent() && // this block is required for uiGetLayers, but not uiGetLayersA/AP
            uidO.get().isPublic() // also, there is no need to show possible layers for PDD & YT
        ) {
            // Show possible layers
            SetF<PassportSid> sids = lwrs.filterMap(l ->
                    Option.when(LayerRoutines.isService(l.getLayer()), l.getLayer().getSid())).unique();
            dbSvcRoutines.addPossibleLayers(sids, ctx.getRootElement());
        }
    }

}
