package ru.yandex.calendar.frontend.web.cmd.run.ui.resource;

import javax.annotation.Nullable;

import lombok.val;
import org.jdom.Element;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.webNew.ResourceComparators;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.AvailabilityInterval;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervals;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.avail.SubjectId;
import ru.yandex.calendar.logic.event.dao.EventLayerDao;
import ru.yandex.calendar.logic.event.dao.EventUserDao;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.schedule.EventScheduleInfoXmlizer;
import ru.yandex.calendar.logic.resource.schedule.ResourceEventsAndReservations;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeFormatter;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;

/**
 * @author gutman
 * @author shinderuk
 */
public class CmdGetDomainResourcesWithScheduleLite extends CmdGetDomainResourcesBase {

    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private EventScheduleInfoXmlizer eventScheduleInfoXmlizer;
    @Autowired
    private EventUserDao eventUserDao;
    @Autowired
    private EventLayerDao eventLayerDao;
    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    private final String day;
    private final Language lang;
    private final ViewType viewType;

    public CmdGetDomainResourcesWithScheduleLite(
            AuthInfo ai, String day, @Nullable String officeId, @Nullable String viewType, @Nullable String lang)
    {
        super("resources-with-schedule-and-offices-lite", ai, officeId);
        this.day = day;
        this.lang = Language.fromValueSafe(lang).getOrElse(Language.RUSSIAN);
        this.viewType = ViewType.fromValueSafe(viewType).getOrElse(ViewType.DAY);
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        PassportDomain domain = getPassportDomain();
        long officeId = officeOrUserDefaultOfficeOrDomainDefaultOffice(domain);
        final PassportUid uid = uidO.get();

        final DateTimeZone userTimezone = dateTimeManager.getTimeZoneForUid(uid);
        final DateTimeZone timeZone = officeManager.getTimeZoneByOfficeId(officeId);

        LocalDate localDate;
        if (StringUtils.isEmpty(day)) {
            localDate = new LocalDate(timeZone);
        } else {
            localDate = TimeUtils.localDate.parse(day);
        }

        final Instant endOfDay = localDate.plusDays(1).toDateTimeAtStartOfDay(timeZone).toInstant();

        int officeToUserTzOffsetHours =
                (timeZone.getOffset(endOfDay) - userTimezone.getOffset(endOfDay))
                / (1000 * 60 * 60);

        CalendarXmlizer.setDtfAttr(ctx.getRootElement(), "now", now, timeZone);
        CalendarXmlizer.setAttr(ctx.getRootElement(), "now-tz-offset", timeZone.getOffset(now));

        CalendarXmlizer.setAttr(ctx.getRootElement(), "tz", timeZone.getID());
        CalendarXmlizer.setAttr(ctx.getRootElement(), "tz-offset", officeToUserTzOffsetHours);

        layerRoutines.getOrCreateDefaultLayer(uid);
        CalendarXmlizer.setAttr(ctx.getRootElement(), "can-open-event", true);

        ctx.getRootElement().addContent(officesXml(lang));

        Comparator<ResourceInfo> resourceComparator = ResourceInfo.resourceF().andThen(
                ResourceFields.FLOOR_NUM.getF().andThenNaturalComparator().thenComparing(
                        ResourceComparators.BY_GROUP_NAME.uncheckedCastC()).thenComparing(
                        ResourceFields.POS.getF().andThenNaturalComparator()).thenComparing(
                        ResourceFields.NAME.getF().andThenNaturalComparator()).uncheckedCastC());

        DateTime.Property rounding =
                localDate.toDateTime(new LocalTime(12, 0), timeZone).property(viewType.getDateTimeFieldType());
        InstantInterval scheduleInterval = new InstantInterval(rounding.roundFloorCopy(), rounding.roundCeilingCopy());

        ListF<ResourceInfo> resourceInfos = resourceRoutines.getDomainResourcesCanViewWithLayersAndOffices(
                uid, OfficeFilter.byId(officeId), ResourceFilter.any());

        final Tuple2List<ResourceInfo, ResourceEventsAndReservations> schedules = resourceScheduleManager
                .getResourceScheduleDataForInterval(
                        uidO, resourceInfos, scheduleInterval, timeZone, Option.empty(), getActionInfo())
                .sortedBy1(resourceComparator);

        ListF<Long> eIds = schedules.get2().flatMap(ResourceEventsAndReservations::getEventIds).stableUnique();

        val userLayerIds = layerRoutines.getLayerIdsByUid(uid);
        ListF<Long> userLayersEventIds = eventLayerDao.findEventLayerEventIdsByLayerIdsAndEventIds(userLayerIds, eIds);
        ListF<EventUser> userEventUsers = eventUserDao.findEventUsersByEventIdsAndUid(userLayersEventIds, uid)
                .filter(EventUser.getDecisionF().andThenEquals(Decision.NO).notF());

        SetF<Long> userEventIds = userEventUsers.map(EventUser.getEventIdF()).unique();

        MapF<PassportUid, String> loginByUid = Cf.map();

        if (passportAuthDomainsHolder.containsYandexTeamRu()) {
            loginByUid = schedules.get2()
                    .flatMap(ResourceEventsAndReservations::getReservationCreatorUids)
                    .zipWithFlatMapO(userManager::getYtUserLoginByUid).toMap();
        }

        // resource schedules
        Element resources = new Element("resources-with-schedule");
        resources.setAttribute("date", localDate.toString());
        for (Tuple2<ResourceInfo, ResourceEventsAndReservations> schedule : schedules) {
            Element element = eventScheduleInfoXmlizer.xmlizeData(
                    userInfoO.get(), schedule, scheduleInterval, timeZone, lang, userEventIds, loginByUid);

            resources.addContent(element);
        }
        ctx.getRootElement().addContent(resources);

        // current user availability
        InstantInterval dayInterval = AuxDateTime.getDayInterval(localDate, timeZone);

        AvailabilityRequest request = AvailabilityRequest.interval(dayInterval).includeEventsNamesWithoutPermsCheck();
        AvailabilityIntervals availabilities = availRoutines.getAvailabilityIntervals(
                uid, SubjectId.uid(uid), request, getActionInfo()).getIntervals();

        Element availability = new Element("availability");
        for (AvailabilityInterval interval : availabilities.unmerged()) {
            Element intervalElement = new Element("interval");
            intervalElement.setAttribute("start-ts",
                    DateTimeFormatter.formatForMachines(interval.getInterval().getStart(), timeZone));
            intervalElement.setAttribute("end-ts",
                    DateTimeFormatter.formatForMachines(interval.getInterval().getEnd(), timeZone));
            intervalElement.setAttribute("availability", interval.getAvailability().toDbValue());
            intervalElement.setAttribute("event-name", interval.getEventName().get());
            availability.addContent(intervalElement);
        }
        ctx.getRootElement().addContent(availability);

    }

}
