package ru.yandex.calendar.frontend.web.cmd.run.ui.resource;

import javax.annotation.Nullable;

import lombok.val;
import org.jdom.Element;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function2B;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.UserXmlCommand;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.resource.schedule.EventIntervalInfo;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleInfo;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.TimeUtils;

public class CmdGetMassageSchedule extends UserXmlCommand {
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;

    private final Option<String> dayStr;
    private final Option<String> exchangeNameStr;

    public CmdGetMassageSchedule(AuthInfo ai, @Nullable String day, @Nullable String exchangeName) {
        super("massage-schedule", ai);
        this.dayStr = StringUtils.notEmptyO(day);
        this.exchangeNameStr = StringUtils.notEmptyO(exchangeName);
    }

    @Override
    protected void buildXmlResponseU(XmlCmdContext ctx) {
        ListF<ResourceInfo> availableResources = resourceRoutines
                .findActiveResources(Cf.list(ResourceType.MASSAGE_ROOM))
                .map(Resource.getIdF().andThen(resourceRoutines.getResourceInfoByIdF()));

        availableResources = availableResources.sorted(ResourceInfo.officeIdF().andThenNaturalComparator()
                .thenComparing(Comparator.naturalComparator().compose(r -> r.getFloorNum().getOrNull()))
                .thenComparing(Comparator.naturalComparator().compose(r -> r.getResource().getPos().getOrNull()))
                .thenComparing(Comparator.naturalComparator().compose(r -> r.getName().getOrNull())));

        Function2B<ResourceInfo, Option<String>> exchangeNameIsInF =
                (r, n) -> r.getResource().getExchangeName().exists(n.containsF());

        ResourceInfo resource = availableResources.find(exchangeNameIsInF.bind2(exchangeNameStr))
                .orElse(availableResources.find(exchangeNameIsInF.bind2(Option.of("conf_spb_massage"))))
                .getOrElse(availableResources.first());

        DateTimeZone timeZone = OfficeManager.getOfficeTimeZone(resource.getOffice());
        LocalDate date = dayStr.isPresent() ? TimeUtils.localDate.parse(dayStr.get()) : new LocalDate(timeZone);

        Element rootEl = ctx.getRootElement();
        rootEl.setAttribute("show-date", date.toString());

        Element resourcesEl = CalendarXmlizer.appendElm(rootEl, "resources");
        for (ResourceInfo cur : availableResources) {
            Element resourceEl = CalendarXmlizer.appendElm(resourcesEl, "resource");

            cur.getResource().appendXmlTo(resourceEl, timeZone, ResourceFields.EXCHANGE_NAME, ResourceFields.NAME);

            CalendarXmlizer.appendElm(resourceEl, "office-name", cur.getOffice().getName());
            CalendarXmlizer.appendElm(resourceEl, "selected", cur.getResourceId() == resource.getResourceId());

            CalendarXmlizer.appendElm(resourceEl, "exchange-name", cur.getResource().getExchangeName().getOrElse("?"));
        }

        ResourceScheduleInfo schedule = resourceScheduleManager.getResourceScheduleForDayAndResource(
                uidO.get(), date, timeZone, resource, getActionInfo().getActionSource());

        Element eventsEl = CalendarXmlizer.appendElm(rootEl, "events");
        int idx = 0;
        for (EventIntervalInfo event : schedule.getEvents()) {
            Option<String> userName = Option.empty();
            Option<Integer> userPhone = Option.empty();

            Participants participants = event.getEventWithRelations().getParticipants();
            Option<YandexUserParticipantInfo> userO = participants.getLastYandexUser();
            if (userO.isPresent()) {
                userName = userO.get().getSettings().getUserName();

                Option<String> userLogin = userO.get().getSettings().getUserLogin();
                if (StringUtils.isBlank(userName.getOrElse(""))) {
                    userName = userLogin.orElse(Option.of("?"));
                }

                if (userLogin.isPresent()) {
                    val userDetails = userManager.getYtUserByLogin(userLogin.get());
                    if (userDetails.isPresent()) {
                        userPhone = Option.x(userDetails.get().getInfo().getWorkPhone().map(Long::intValue));
                    }
                }
            }
            Element eventEl = CalendarXmlizer.appendElm(eventsEl, "event");
            CalendarXmlizer.setAttr(eventEl, "index", ++idx);
            CalendarXmlizer.appendDtfElm(eventEl, "start", event.getInterval().getStart(), timeZone);
            CalendarXmlizer.appendDtfElm(eventEl, "end", event.getInterval().getEnd(), timeZone);
            CalendarXmlizer.appendElm(eventEl, "name", event.getEventWithRelations().getEvent().getName());
            CalendarXmlizer.appendElm(eventEl, "user-name", userName.getOrElse(""));
            CalendarXmlizer.appendElm(eventEl, "user-phone", userPhone.map(Cf.Integer.toStringF()).getOrElse(""));
        }
    }
}
