package ru.yandex.calendar.frontend.web.cmd.run.ui.resource;

import org.jdom.Element;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.util.xml.CalendarXmlizer;

/**
 * @author Stepan Koltsov
 */
public class ResourceXmlizer {

    public static Element toXml(ResourceInfo resourceInfo) {
        return toXml(resourceInfo, Language.RUSSIAN);
    }

    public static Element toXml(ResourceInfo resourceInfo, Language lang) {
        return toXml(resourceInfo, Option.empty(), lang);
    }

    public static Element toXml(ResourceInfo resourceInfo, Option<NameI18n> protection, Language lang) {
        Element element = new Element("resource");
        resourceInfo.getResource().appendXmlTo(element, DateTimeZone.UTC,
                ResourceFields.ID,
                ResourceFields.DESCRIPTION,
                ResourceFields.EXCHANGE_NAME,
                ResourceFields.FLOOR_NUM,
                ResourceFields.POS,
                ResourceFields.SEATS,
                ResourceFields.CAPACITY,
                ResourceFields.VIDEO,
                ResourceFields.PHONE,
                ResourceFields.VOICE_CONFERENCING,
                ResourceFields.PROJECTOR,
                ResourceFields.LCD_PANEL,
                ResourceFields.MARKER_BOARD,
                ResourceFields.DESK,
                ResourceFields.GUEST_WIFI,
                ResourceFields.MAP_URL,
                ResourceFields.IS_ACTIVE
        );
        CalendarXmlizer.appendElm(element, "type", protection.isPresent()
                ? ResourceType.PROTECTED_ROOM
                : resourceInfo.getResource().getType());

        CalendarXmlizer.appendElm(element, "name", resourceInfo.getNameI18n(lang).getOrNull());
        CalendarXmlizer.appendElm(element, "alter-name", resourceInfo.getAlterNameI18n(lang).getOrNull());
        CalendarXmlizer.appendElm(element, "office-id", officeIdForVerstka(resourceInfo.getOffice()));
        CalendarXmlizer.appendElm(element, "office-name", resourceInfo.getOfficeNameI18n(lang));
        CalendarXmlizer.appendElm(element, "city-name", resourceInfo.getCityNameI18n(lang).getOrNull());
        CalendarXmlizer.appendElm(element, "group-name", resourceInfo.getGroupNameI18n(lang).getOrNull());

        protection.map(n -> n.getName(lang))
                .orElse(ResourceRoutines.getProtectionMessageI18n(resourceInfo.getResource(), lang))
                .forEach(s -> CalendarXmlizer.appendElm(element, "protection-message", s));
        return element;
    }

    public static ListF<Element> officeXmls(ListF<Office> offices, Language lang) {
        return offices.zip(offices)
                .map1(officeNameForVerstkaF(lang)).map2(officeIdForVerstkaF())
                .sortedBy1()
                .map(officeXmlF());
    }

    private static Function2<String, String, Element> officeXmlF() {
        return new Function2<String, String, Element>() {
            public Element apply(String name, String id) {
                Element e = new Element("office");
                CalendarXmlizer.appendElm(e, "name", name);
                CalendarXmlizer.appendElm(e, "id", id);
                return e;
            }
        };
    }

    private static Function<Office, String> officeNameForVerstkaF(final Language lang) {
        return new Function<Office, String>() {
            public String apply(Office office) {
                return getOfficeNameForVerstka(office, lang);
            }
        };
    }

    public static String getOfficeNameForVerstka(Office office, Language lang) {
        String officeName = ResourceRoutines.getNameI18n(office, lang);
        Option<String> cityName = ResourceRoutines.getCityNameI18n(office, lang);

        if (cityName.isSome(officeName)) {
            return officeName;
        }
        return cityName.plus1(officeName).mkString(", ");
    }

    private static Function<Office, String> officeIdForVerstkaF() {
        return new Function<Office, String>() {
            public String apply(Office office) {
                return officeIdForVerstka(office);
            }
        };
    }

    static String officeIdForVerstka(Office office) {
        return String.valueOf(office.getId());
    }

} //~
