package ru.yandex.calendar.frontend.web.cmd.run.ui.todo;

import org.jdom.Element;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.web.AuthInfo;
import ru.yandex.calendar.frontend.web.cmd.ctx.XmlCmdContext;
import ru.yandex.calendar.frontend.web.cmd.generic.ValidatableXmlCommand;
import ru.yandex.calendar.logic.beans.generated.SettingsFields;
import ru.yandex.calendar.logic.todo.TodoListsXmlAppender;
import ru.yandex.calendar.logic.todo.TodoListsXmlAppender.Mode;
import ru.yandex.calendar.logic.todo.TodoViewType;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserDao;
import ru.yandex.calendar.util.validation.RequestValidator;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder.PolicyHandle;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 */
public class CmdGetTodoLists extends ValidatableXmlCommand {
    private static final Logger logger = LoggerFactory.getLogger(CmdGetTodoLists.class);
    private static final String CMD_TAG = "get-todo-lists";

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private TodoListsXmlAppender todoListsXmlAppender;
    @Autowired
    private UserDao userDao;

    private final String todoViewTypeStr;
    private final Long todoListId;

    public CmdGetTodoLists(AuthInfo ai, String todoViewType) {
        super(CMD_TAG, ai);
        this.todoViewTypeStr = todoViewType;
        this.todoListId = null;
    }

    public CmdGetTodoLists(
            AuthInfo ai,
            String todoViewType, long todoListId) {
        super(CMD_TAG, ai);
        this.todoViewTypeStr = todoViewType;
        this.todoListId = todoListId;
    }

    @Override
    public void validate() {
        RequestValidator.validateOptional(RequestValidator.POSITIVE, "todoListId", todoListId);
    }

    @Override
    protected void buildXmlResponseV(XmlCmdContext ctx) {
        Element rootElement = ctx.getRootElement();
        logger.debug(("View type mode: " + todoViewTypeStr));

        TodoViewType viewType;
        if (StringUtils.isEmpty(todoViewTypeStr)) {
            viewType = settingsRoutines.getSettingsField(uidO.get(), SettingsFields.TODO_VIEW_TYPE);
        } else {
            viewType = TodoViewType.find(todoViewTypeStr);
            try {
                PolicyHandle h = MasterSlaveContextHolder.push(MasterSlavePolicy.RW_M);
                try {
                    userDao.updateTodoViewTypeByUid(uidO.get(), viewType);
                } finally {
                    h.popSafely();
                }
            } catch (Exception e) {
                // XXX: pass error to verstka as warning
                logger.warn("Failed to update view: " + e, e);
            }
        }
        CalendarXmlizer.appendElm(rootElement, "todo-view-type", viewType.toDbValue());
        Mode mode = viewType == TodoViewType.ALL ? TodoListsXmlAppender.Mode.ALL : TodoListsXmlAppender.Mode.DEFAULT;
        SqlCondition sqlCondition = viewType.getSearchCondition(tz,
                settingsRoutines.getSettingsField(uidO.get(), SettingsFields.START_WEEKDAY));
        todoListsXmlAppender.appendXml(rootElement, uidO.get(), tz, mode, Option.ofNullable(todoListId), sqlCondition);
    }
}
