package ru.yandex.calendar.frontend.webNew;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.frontend.webNew.suggest.OfficeFloor;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.algo.AlphanumericComparator;
import ru.yandex.misc.lang.Validate;

/**
 * @author Daniel Brylev
 */
public class ResourceComparators {

    public static final Comparator<Resource> BY_GROUP_NAME = ResourceFields.GROUP_NAME.getF()
            .andThen(Comparator.wrap(AlphanumericComparator.SENSITIVE).nullLowC()).uncheckedCastC();

    @Autowired
    private SettingsRoutines settingsRoutines;

    public Comparator<ResourceInfo> distanceFromSelectedResourcesComparator(ListF<ResourceInfo> selectedResources) {
        return selectedResources.isNotEmpty()
                ? new SelectedResourcesDistanceComparator(selectedResources)
                : Comparator.<ResourceInfo>constEqualComparator();
    }

    public Comparator<ResourceInfo> distanceFromUserComparator(PassportUid uid) {
        Option<SettingsYt> settingsO = settingsRoutines.getSettingsByUidIfExists(uid).filterMap(SettingsInfo.getYtF());

        Option<Long> officeIdO = settingsO.filterMap(SettingsYt.getActiveOfficeIdF());
        Option<Long> tableOfficeIdO = settingsO.filterMap(SettingsYt.getTableOfficeIdF());

        Option<Integer> floorNumO = Option.when(tableOfficeIdO.equals(officeIdO),
                settingsO.filterMap(SettingsYt.getTableFloorNumF())).<Integer>flatten().singleO();

        return officeIdO.isPresent()
                ? new FloorDistanceComparator(officeIdO.get(), floorNumO)
                : Comparator.<ResourceInfo>constEqualComparator();
    }

    public Comparator<ResourceInfo> distanceFromFloorComparator(OfficeFloor officeFloor) {
        return new FloorDistanceComparator(officeFloor.getOfficeId(), officeFloor.getFloorNum());
    }

    public Comparator<ResourceInfo> hasIdComparator(ListF<Long> ids) {
        return positiveLowComparator(ResourceInfo.resourceIdF().andThen(ids.unique().containsF()));
    }

    public Comparator<ResourceInfo> hasEnoughCapacityComparator(int requiredCapacity) {
        return positiveLowComparator(
                ResourceInfo.getCapacityF().andThen(Cf2.f(us -> us.getOrElse(10))).andThen(Cf.Integer.geF(requiredCapacity)));
    }

    public Comparator<ResourceInfo> hasVideoConferencingComparator() {
        return positiveLowComparator(ResourceInfo.hasVideoF());
    }

    public Comparator<ResourceInfo> hasAnyPhoneComparator() {
        Function1B<Option<Integer>> isDefinedIntF = Option::isPresent;

        return positiveLowComparator(ResourceInfo.resourceF().andThen(
                Resource.getPhoneF().andThen(isDefinedIntF).orF(Resource.getVideoF().andThen(isDefinedIntF))));
    }

    public static Comparator<ResourceInfo> positiveLowComparator(Function1B<ResourceInfo> predicate) {
        return predicate.asFunction().andThenNaturalComparator().reversed();
    }

    private static class SelectedResourcesDistanceComparator implements Comparator<ResourceInfo> {
        private final MapF<Long, SetF<Long>> selectedResourcesIdsByByOfficeId;
        private final MapF<Long, ListF<Integer>> selectedFloorNumsByOfficeId;

        private SelectedResourcesDistanceComparator(ListF<ResourceInfo> selectedResources) {
            Tuple2List<Long, ListF<Resource>> selectedResourcesByOfficeId = selectedResources
                    .toTuple2List(ResourceInfo.officeIdF(), ResourceInfo.resourceF()).groupBy1().entries();

            this.selectedResourcesIdsByByOfficeId = selectedResourcesByOfficeId
                    .map2(rs -> rs.map(Resource::getId).unique()).toMap();
            this.selectedFloorNumsByOfficeId = selectedResourcesByOfficeId
                    .map2(rs -> rs.filterMap(Resource::getFloorNum)).toMap();
        }

        @Override
        public int compare(ResourceInfo resource1, ResourceInfo resource2) {
            Validate.equals(resource1.getOfficeId(), resource2.getOfficeId(),
                    "can not compare resources from different offices");

            final long officeId = resource1.getOfficeId();
            SetF<Long> selectedIds = selectedResourcesIdsByByOfficeId.getOrElse(officeId, Cf.<Long>set());
            ListF<Integer> selectedFloorNums = selectedFloorNumsByOfficeId.getOrElse(officeId, Cf.<Integer>list());

            if (selectedIds.containsTs(resource1.getResourceId()) || selectedIds.containsTs(resource2.getResourceId())) {
                if (!selectedIds.containsTs(resource2.getResourceId())) return -1;
                if (!selectedIds.containsTs(resource1.getResourceId())) return 1;
                return 0;
            }
            if (selectedFloorNums.isEmpty()) return 0;

            if (resource1.getFloorNum().isPresent() != resource2.getFloorNum().isPresent()) {
                return resource1.getFloorNum().isPresent() ? -1 : 1;
            }
            if (resource1.getFloorNum().isPresent() && resource2.getFloorNum().isPresent()) {
                int distance1 = distance(resource1.getFloorNum().get(), selectedFloorNums);
                int distance2 = distance(resource2.getFloorNum().get(), selectedFloorNums);
                return distance1 - distance2;
            }
            return 0;
        }

        private int distance(int resourceFloorNum, ListF<Integer> selectedFloorNums) {
            return Math.abs(selectedFloorNums.map(Cf.Integer.minusF().bind1(resourceFloorNum)).min());
        }
    }

    private static class FloorDistanceComparator implements Comparator<ResourceInfo> {
        private final long officeId;
        private final Option<Integer> floorNum;

        private FloorDistanceComparator(long officeId, Option<Integer> floorNum) {
            this.officeId = officeId;
            this.floorNum = floorNum;
        }

        @Override
        public int compare(ResourceInfo resource1, ResourceInfo resource2) {
            if (resource1.getOfficeId() != officeId || resource2.getOfficeId() != officeId) {
                if (resource1.getOfficeId() == officeId) return -1;
                if (resource2.getOfficeId() == officeId) return 1;
                return 0;
            }
            if (!floorNum.isPresent()) return 0;

            if (resource1.getFloorNum().isPresent() != resource2.getFloorNum().isPresent()) {
                return resource1.getFloorNum().isPresent() ? -1 : 1;
            }
            if (resource1.getFloorNum().isPresent() && resource2.getFloorNum().isPresent()) {
                return distance(resource1.getFloorNum().get()) - distance(resource2.getFloorNum().get());
            }
            return 0;
        }

        private int distance(int resourceFloorNum) {
            return Math.abs(floorNum.get() - resourceFloorNum);
        }
    }
}
