package ru.yandex.calendar.frontend.webNew;

import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.Lazy;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.bender.WebDate;
import ru.yandex.calendar.frontend.web.cmd.run.ui.CmdGetHolidaysA;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailDisplayMode;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailParameters;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailShapeType;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailabilitiesData;
import ru.yandex.calendar.frontend.webNew.dto.in.SuggestData;
import ru.yandex.calendar.frontend.webNew.dto.out.HolidaysInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.LocalDateTimeInterval;
import ru.yandex.calendar.frontend.webNew.dto.out.ReservationInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ResourcesSchedule;
import ru.yandex.calendar.frontend.webNew.dto.out.SubjectAvailability;
import ru.yandex.calendar.frontend.webNew.dto.out.SubjectAvailabilityIntervals;
import ru.yandex.calendar.frontend.webNew.dto.out.SubjectsAvailabilities;
import ru.yandex.calendar.frontend.webNew.dto.out.SubjectsAvailabilityIntervals;
import ru.yandex.calendar.frontend.webNew.dto.out.SuggestDatesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.SuggestInfo;
import ru.yandex.calendar.frontend.webNew.suggest.OfficeFloor;
import ru.yandex.calendar.frontend.webNew.suggest.OfficeUsers;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventChangesFinder;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.InaccessibleResourcesRequest;
import ru.yandex.calendar.logic.event.avail.AvailRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervals;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervalsOrRefusal;
import ru.yandex.calendar.logic.event.avail.AvailabilityOverlap;
import ru.yandex.calendar.logic.event.avail.AvailabilityOverlapOrRefusal;
import ru.yandex.calendar.logic.event.avail.AvailabilityQueryRefusalReason;
import ru.yandex.calendar.logic.event.avail.AvailabilityRequest;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.EventResourceDao;
import ru.yandex.calendar.logic.event.repetition.RecurrenceTimeInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceSet;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.logic.resource.ResourceInaccessibility;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.UidOrResourceId;
import ru.yandex.calendar.logic.resource.reservation.ResourceReservationManager;
import ru.yandex.calendar.logic.suggest.AvailableResourcesInOffices;
import ru.yandex.calendar.logic.suggest.OfficeAndFreeResources;
import ru.yandex.calendar.logic.suggest.SearchInterval;
import ru.yandex.calendar.logic.suggest.Suggest;
import ru.yandex.calendar.logic.suggest.SuggestManager;
import ru.yandex.calendar.logic.suggest.SuggestedResources;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.holidays.DayInfo;
import ru.yandex.commune.holidays.DayInfoHandler;
import ru.yandex.commune.holidays.HolidayRoutines;
import ru.yandex.commune.holidays.OutputMode;
import ru.yandex.inside.geobase.GeobaseIds;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

public class WebNewAvailabilityManager {
    private static final int MAX_RESOURCE_OPTIONS_PER_OFFICE = 3;
    private static final Duration SUGGEST_SEARCH_DURATION = Duration.standardDays(SuggestManager.SUGGEST_SEARCH_DAYS);

    @Autowired
    private AvailRoutines availRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private SuggestManager suggestManager;
    @Autowired
    private ResourceComparators resourceComparators;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private ResourceReservationManager resourceReservationManager;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private EventResourceDao eventResourceDao;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private WebNewResourcesManager webNewResourcesManager;
    @Autowired
    private EventDbManager eventDbManager;

    public SuggestInfo suggest(
            PassportUid uid, SuggestData suggestData, AvailParameters params, Language lang)
    {
        ValidateParam.notEmpty("offices", suggestData.getOffices(), "Empty offices, nothing to suggest");
        ValidateParam.isTrue("data", !suggestData.getSearchBackward().isPresent() || suggestData.getSearchStart().isPresent());
        ValidateParam.isTrue("interval", !suggestData.getEventStart().isAfter(suggestData.getEventEnd()));

        val knownUsers = userManager.getFlatUidValuesByEmailsSafe(suggestData.getUsers());

        if (knownUsers.isEmpty()) {
            return SuggestInfo.empty();
        }

        DateTimeZone tz = params.tz.getOrElse(dateTimeManager.getTimeZoneForUidF().bind(uid));

        Suggest suggest;

        if (!suggestData.getRepetitionData().isPresent()) {
            SearchInterval interval = getSearchInterval(suggestData, tz);
            suggest = suggestManager.singleMeetingSuggest(
                    uid, Cf.list(knownUsers), suggestData, interval, params.exceptEventId, tz, getActionInfo());

            suggest = new Suggest(suggest.getSuggestedIntervals(),
                    suggest.getBackSearchStart().filter(getActionInfo().getNow()::isBefore),
                    suggest.getNextSearchStart());

        } else {
            Repetition repetition = suggestData.getRepetition(tz).get();
            DateTimeZone repeatTz = eventWebManager.getEventTimezoneIfExists(params.exceptEventId).getOrElse(tz);

            suggest = suggestManager.repeatingMeetingSuggest(
                    uid, Cf.list(knownUsers), suggestData, repetition, repeatTz, params.exceptEventId, tz, getActionInfo());
        }
        if (suggestData.isSameRoomMode()) {
            return toSuggestInfoWithNoPlaces(uid, suggestData, suggest, tz);

        } else {
            MapF<Long, Comparator<ResourceInfo>> comps = getComparatorsForRequestedOffices(uid, Cf.list(knownUsers), suggestData);
            return toSuggestInfoWithPlaces(userManager.getUserInfo(uid), suggestData, suggest, comps, tz, lang);
        }
    }

    public ResourcesSchedule suggestResources(
            PassportUid uid, SuggestData suggestData, AvailParameters params, Language lang)
    {
        ValidateParam.notEmpty("offices", suggestData.getOffices(), "Empty offices, nothing to suggest");
        ValidateParam.some("date", suggestData.getDate());
        ValidateParam.isTrue("interval", !suggestData.getEventStart().isAfter(suggestData.getEventEnd()));

        UserInfo userInfo = userManager.getUserInfo(uid);
        val knownUsers = userManager.getFlatUidValuesByEmailsSafe(suggestData.getUsers());

        DateTimeZone tz = params.tz.getOrElse(() -> dateTimeManager.getTimeZoneForUid(uid));

        InstantInterval interval = AuxDateTime.getDayInterval(suggestData.getDate().get(), tz);

        SuggestedResources suggest = suggestManager.suggestResources(
                uid, Cf.list(knownUsers), suggestData, tz, params.exceptEventId,
                getComparatorsForRequestedOffices(uid, Cf.list(knownUsers), suggestData), getActionInfo());

        return new ResourcesSchedule(suggest.getOffices().map(o -> new ResourcesSchedule.Office(
                o.getOffice().getId(), o.getOffice().getStaffId(),
                ResourceRoutines.getNameI18n(o.getOffice(), lang),
                OfficeManager.getOfficeTimeZone(o.getOffice()).getID(),
                o.getResources().map(rs -> webNewResourcesManager.toResourceScheduleResource(
                        userInfo, rs.getResource(), rs.getBusyTimes(), interval, true, tz, lang)))));
    }

    public SuggestDatesInfo suggestDates(
            PassportUid uid, SuggestData suggestData, AvailParameters params)
    {
        ValidateParam.notEmpty("offices", suggestData.getOffices(), "Empty offices, nothing to suggest");
        ValidateParam.some("date", suggestData.getDate());
        ValidateParam.isTrue("interval", !suggestData.getEventStart().isAfter(suggestData.getEventEnd()));

        val knownUsers = userManager.getFlatUidValuesByEmailsSafe(suggestData.getUsers());
        DateTimeZone tz = params.tz.getOrElse(() -> dateTimeManager.getTimeZoneForUid(uid));

        return SuggestDatesInfo.of(suggestManager.suggestDates(
                uid, Cf.list(knownUsers), suggestData, tz, params.exceptEventId, getActionInfo()));
    }

    public SubjectsAvailabilityIntervals getAvailabilityIntervals(
            PassportUid uid, ListF<Email> emails,
            Option<LocalDate> date, Option<WebDate> from, Option<WebDate> till,
            AvailDisplayMode display, AvailShapeType shape, AvailParameters params, Language lang)
    {
        ValidateParam.isTrue("dates", from.isPresent() && till.isPresent() || date.isPresent(), "Specify either date or from and till");

        DateTimeZone tz = params.tz.getOrElse(dateTimeManager.getTimeZoneForUidF().bind(uid));

        Instant since = from.orElse(date.map(WebDate::localDate)).get().toInstant(tz);
        Instant until = till.orElse(date.map(WebDate::localDate)).get().toInstantNextDay(tz);

        AvailabilityRequest req = AvailabilityRequest.interval(since, until)
                .withUseResourceCheduleCache(true)
                .excludeEventId(params.exceptEventId);

        req = display.applyTo(req);
        req = shape.applyTo(req);

        return new SubjectsAvailabilityIntervals(getAvailabilityResponses(uid, emails, req,
                SubjectAvailabilityIntervals.intervalsF(shape, tz, lang), SubjectAvailabilityIntervals.refusalF()));
    }

    public SubjectsAvailabilities getAvailabilities(
            PassportUid uid, AvailabilitiesData data, AvailParameters params)
    {
        DateTimeZone tz = params.tz.getOrElse(dateTimeManager.getTimeZoneForUidF().bind(uid));
        DateTimeZone repeatTz = eventWebManager.getEventTimezoneIfExists(params.exceptEventId).getOrElse(tz);

        RepetitionInstanceInfo repetitionInfo = data.toRepetitionInstanceInfo(tz, repeatTz);

        ListF<Email> notActiveEmails = resourceRoutines.selectNotActiveResourceEmails(data.getEmails());
        ListF<SubjectAvailability> notActiveAvailabilities = notActiveEmails.map(SubjectAvailability.notActiveF());

        ListF<Email> activeEmails = data.getEmails().filter(notActiveEmails.containsF().notF());

        ListF<SubjectAvailability> activeAvailabilities = getAvailabilities(
                uid, activeEmails, Either.left(repetitionInfo), params, true, tz);

        return new SubjectsAvailabilities(activeAvailabilities.plus(notActiveAvailabilities));
    }

    public ReservationInfo reserveResources(
            PassportUid uid, AvailabilitiesData data,
            long reservationId, Option<Boolean> keepIfAnyBusy, Option<Email> keepIfResourceBusy, AvailParameters params)
    {
        DateTimeZone tz = params.tz.getOrElse(dateTimeManager.getTimeZoneForUidF().bind(uid));
        DateTimeZone repeatTz = eventWebManager.getEventTimezoneIfExists(params.exceptEventId).getOrElse(tz);

        Tuple2List<Email, Option<Resource>> resourcesEmailIds = resourceRoutines.findResourcesByEmails(data.getEmails());

        Tuple2List<Email, Resource> emailResources = Cf2.flatBy2(resourcesEmailIds);
        ListF<Email> activeResourceEmails = emailResources.filterBy2(Resource::getIsActive).get1();
        ListF<Email> notActiveResourceEmails = emailResources.filterBy2(r -> !r.getIsActive()).get1();

        MapF<Email, Long> idByEmail = emailResources.map2(Resource.getIdF()).toMap();

        Option<Long> keepIfBusyId = keepIfResourceBusy.filterMap(idByEmail::getO)
                .orElse(() -> keepIfResourceBusy.filterMap(resourceRoutines::getResourceId));

        ListF<SubjectAvailability> inactiveAvailabilities = data.getEmails()
                .filterNot(idByEmail::containsKeyTs).map(SubjectAvailability.unknownF())
                .plus(notActiveResourceEmails.map(SubjectAvailability.notActiveF()));

        return transactionTemplate.execute(s -> {
            resourceRoutines.lockResourcesByIds(activeResourceEmails.map(idByEmail::getOrThrow));

            ExistingResourcesRepetitionInfo repetitionResources = findExistingResourcesRepetitionInfo(
                    data.toRepetitionInstanceInfo(tz, repeatTz), emailResources.map(t -> t.get2().getId()), params);

            RepetitionInstanceInfo repetitionInfo = repetitionResources.getRepetitionInfo();

            ListF<SubjectAvailability> availabilities = getAvailabilities(
                    uid, activeResourceEmails, Either.right(repetitionResources), params, false, tz);

            Function1B<SubjectAvailability> isUnavailableF = SubjectAvailability.isAvailableF().notF();
            Function<SubjectAvailability, Long> idF = a -> idByEmail.getOrThrow(a.getEmail());

            ListF<Long> reserveResourceIds = availabilities.filter(isUnavailableF.notF()).map(idF)
                    .filterNot(repetitionResources.getTimeUnchangedResourceIds()::containsTs);

            Option<Instant> minDueTs = repetitionInfo.getUntilDate()
                    .plus(availabilities.filterMap(SubjectAvailability::getDueDate)).minO()
                    .map(d -> EventRoutines.calculateDueTsFromUntilDate(repetitionInfo.getEventStart(), d, repeatTz));

            if (!(keepIfAnyBusy.isSome(true) && availabilities.exists(isUnavailableF))
                && !availabilities.exists(isUnavailableF.andF(idF.andThen(keepIfBusyId.containsF()))))
            {
                resourceReservationManager.createOrUpdateReservations(
                        uid, reservationId, reserveResourceIds,
                        repetitionInfo.withDueTs(minDueTs)
                                .withInstancesInsteadOfTimeUnchangedRecurrences()
                                .withoutPastRecurrencesAndExdates(getActionInfo().getNow()), getActionInfo());
            }
            Option<NameI18n> readable = availabilities.iterator()
                    .filterMap(a -> a.getInaccessibility().map(ResourceInaccessibility::getReadableMessage)).nextO();

            return new ReservationInfo(new SubjectsAvailabilities(availabilities.plus(inactiveAvailabilities)), readable);
        });
    }

    public void cancelResourcesReservation(PassportUid uid, long reservationId) {
        resourceReservationManager.cancelReservations(uid, reservationId);
    }

    public HolidaysInfo getHolidays(
            Option<PassportUid> uid, LocalDate from, LocalDate to, String holidaysFor, Option<OutputMode> outModeO)
    {
        Option<SettingsYt> settingsYt = uid.filter(PassportUid::isYandexTeamRu)
                .map(settingsRoutines::getSettingsByUid).filterMap(SettingsInfo.getYtF());

        int countryId = settingsYt.isPresent() && settingsYt.get().getTableOfficeId().isPresent()
                ? officeManager.getCountryIdByOfficeId(settingsYt.get().getTableOfficeId().get())
                : CmdGetHolidaysA.lookupGeoId(holidaysFor).orElse(GeobaseIds.RUSSIA);

        OutputMode outMode = outModeO.getOrElse(OutputMode.HOLIDAYS_AND_WITH_NAMES);
        boolean outLabels = CmdGetHolidaysA.isOutLabelsMode(outMode);

        boolean forYandex = passportAuthDomainsHolder.containsYandexTeamRu();

        ListF<HolidaysInfo.Holiday> holidays = Cf.arrayList();

        HolidayRoutines.processDates(from, to, countryId, outMode, forYandex, new DayInfoHandler() {
            public void handle(LocalDate date, DayInfo info) {
                Option<String> name = info.getNames().isNotEmpty() && (outLabels || info.isOverride())
                        ? Option.when(!outLabels, info.getNames().first()).orElse(info.getNameO())
                        : Option.empty();

                holidays.add(new HolidaysInfo.Holiday(date, info.getDayType(), info.getTransferDate(), name));
            }
        });
        return new HolidaysInfo(holidays);
    }

    public ListF<SubjectAvailability> getAvailabilities(
            PassportUid uid, ListF<Email> emails,
            Either<RepetitionInstanceInfo, ExistingResourcesRepetitionInfo> repetitionInfos,
            AvailParameters params, boolean useResourceScheduleCache, DateTimeZone tz)
    {
        if (emails.isEmpty()) return Cf.list();

        ListF<ResourceInfo> resources = resourceRoutines.getDomainResourcesCanViewWithLayersAndOffices(
                uid, OfficeFilter.any(), ResourceFilter.byEmail(emails));

        RepetitionInstanceInfo repetitionInfo = repetitionInfos.fold(r -> r, ExistingResourcesRepetitionInfo::getRepetitionInfo);

        ListF<SubjectAvailability> inaccessibleAvailabilities = eventRoutines
                .findInaccessibleResources(
                        Option.of(userManager.getUserInfo(uid)),
                        InaccessibleResourcesRequest.noEventsLookup(resources, repetitionInfo, getActionInfo().getNow())
                )
                .map(SubjectAvailability::inaccessible);

        emails = emails.filterNot(inaccessibleAvailabilities.map(SubjectAvailability.getEmailF())::containsTs);

        ListF<SubjectAvailability> accessibleAvailabilities = !repetitionInfo.getRepetition().isPresent()
                ? getAvailabilitiesForSingleInterval(
                        uid, emails, repetitionInfo.getEventInterval(), params.exceptEventId, useResourceScheduleCache)
                : getAvailabilitiesForRepeatingInterval(
                        uid, emails, repetitionInfos, params, useResourceScheduleCache, tz);

        return accessibleAvailabilities.plus(inaccessibleAvailabilities);
    }

    public ExistingResourcesRepetitionInfo findExistingResourcesRepetitionInfo(
            RepetitionInstanceInfo newInfo, ListF<Long> newResourceIds, AvailParameters params)
    {
        Option<Long> eventId = newInfo.getRepetition().isPresent()
                ? params.exceptEventId.flatMapO(eventDao::findMasterEventByEventId).map(Event::getId)
                : params.exceptEventId;

        Option<RepetitionInstanceInfo> repetitionO = eventId.flatMapO(eventDbManager::getEventByIdSafe)
                .map(repetitionRoutines::getRepetitionInstanceInfoByEvent);

        Lazy<ListF<Long>> resourceIds = Lazy.withSupplier(() -> eventResourceDao.findEventResourceIdsByEventIds(eventId));

        Repetition repetitionChanges = EventChangesFinder.repetitionChanges(repetitionO, newInfo.getRepetitionOrNone());

        boolean ruleChanged = RepetitionUtils.repetitionRuleChanged(repetitionChanges);

        if (params.instanceStart.isPresent() && repetitionO.isPresent()) {
            boolean timeChanged = !newInfo.getEventStart().equals(params.instanceStart.get())
                    || !newInfo.getEventPeriod().equals(repetitionO.get().getEventPeriod());

            boolean hasActual = repetitionO.get()
                    .hasActualOccurrenceIn(getActionInfo().getNow(), params.instanceStart.get());

            if (timeChanged || ruleChanged || hasActual && !newResourceIds.forAll(resourceIds.get()::containsTs)) {
                return new ExistingResourcesRepetitionInfo(newInfo, Cf.list());
            }
        }
        boolean timeChanged = repetitionO.exists(r -> !r.containsInterval(newInfo.getEventInterval()));

        boolean timeOrRuleChanged = timeChanged || ruleChanged;

        repetitionO = repetitionO.map(r -> r.withDueTs(newInfo.getRepetition().flatMapO(Repetition::getDueTs)));

        return new ExistingResourcesRepetitionInfo(
                timeOrRuleChanged ? newInfo : repetitionO.getOrElse(newInfo),
                !timeChanged && repetitionChanges.cardinality() == 0 ? resourceIds.get() : Cf.list());
    }

    private ListF<SubjectAvailability> getAvailabilitiesForSingleInterval(
            PassportUid uid, ListF<Email> emails, InstantInterval interval,
            Option<Long> exceptEventId, boolean useResourceScheduleCache)
    {
        AvailabilityRequest req = AvailabilityRequest.interval(interval).excludeEventId(exceptEventId)
                .withUseResourceCheduleCache(useResourceScheduleCache);

        Function2<Email, AvailabilityIntervals, SubjectAvailability> successF = SubjectAvailability.knownF().compose2(
                AvailabilityIntervals.getMostBusyAvailabilityF().andThen(Cf2.f(us -> us.getOrElse(Availability.AVAILABLE))));
        Function2<Email, AvailabilityQueryRefusalReason, SubjectAvailability> refusalF =
                Cf2.asFunction2Ignore2(SubjectAvailability.unknownF());

        return getAvailabilityResponses(uid, emails, req, successF, refusalF);
    }

    private ListF<SubjectAvailability> getAvailabilitiesForRepeatingInterval(
            PassportUid uid, ListF<Email> emails,
            Either<RepetitionInstanceInfo, ExistingResourcesRepetitionInfo> repetitionInfos,
            AvailParameters params, boolean useResourceScheduleCache, DateTimeZone tz)
    {
        Tuple2List<Email, Option<UidOrResourceId>> emailsSubjects = eventInvitationManager.getSubjectIdsByEmails(emails);
        ListF<Long> newResourceIds = emailsSubjects.filterMap(t -> t.get2().flatMapO(UidOrResourceId::getResourceIdO));

        ExistingResourcesRepetitionInfo repetitionResources = repetitionInfos.fold(
                r -> findExistingResourcesRepetitionInfo(r, newResourceIds, params), r -> r);

        RepetitionInstanceInfo repetitionInfo = repetitionResources.getRepetitionInfo();

        ListF<Long> timeUnchangedResources = repetitionResources.getTimeUnchangedResourceIds();

        Function1B<UidOrResourceId> isNewResourceF = id -> id.isResource()
                && !timeUnchangedResources.containsTs(id.getResourceId());

        ListF<UidOrResourceId> resourceIds = emailsSubjects.filterMap(t -> t.get2().filter(isNewResourceF));
        ListF<UidOrResourceId> userIds = emailsSubjects.filterMap(t -> t.get2().filter(UidOrResourceId::isUser));

        MapF<UidOrResourceId, Email> emailBySubjectId = Cf2.flatBy2(emailsSubjects).toMap(Tuple2::swap);

        ListF<InstantInterval> timeUnchangedFutureRecurrences = repetitionInfo.getRecurrences()
                .map(RecurrenceTimeInfo::getInterval)
                .filter(i -> i.getEnd().isAfter(getActionInfo().getNow()))
                .filter(repetitionInfo::containsInterval);

        RepetitionInstanceSet futureRepetitions = RepetitionInstanceSet
                .boundedByMaxCheckPeriod(repetitionInfo, getActionInfo().getNow())
                .plusIntervals(timeUnchangedFutureRecurrences);

        AvailabilityRequest request = AvailabilityRequest.intervals(futureRepetitions)
                .excludeEventId(params.exceptEventId)
                .withUseResourceCheduleCache(useResourceScheduleCache);

        ListF<AvailabilityOverlapOrRefusal> resourceOverlaps = availRoutines.findFirstAvailabilityOverlaps(
                uid, resourceIds, request, getActionInfo());

        ListF<SubjectAvailability> resourceAvailabilities = Cf.arrayList();

        timeUnchangedResources.forEach(resourceId -> emailBySubjectId.getO(UidOrResourceId.resource(resourceId))
                .map(email -> resourceAvailabilities.add(SubjectAvailability.known(email, Availability.AVAILABLE))));

        for (AvailabilityOverlapOrRefusal overlap : resourceOverlaps) {
            Email email = emailBySubjectId.getOrThrow(overlap.getSubjectId());

            if (overlap.isRefusal()) {
                resourceAvailabilities.add(SubjectAvailability.unknown(email));

            } else if (!overlap.getOverlap().isPresent() || futureRepetitions.isEmpty()) {
                resourceAvailabilities.add(SubjectAvailability.known(email, Availability.AVAILABLE));

            } else {
                Instant overlappingInstanceStart = overlap.getOverlap().get().getInstanceInterval().getStart();
                int freeRepetitionsInFuture = futureRepetitions.countInstancesEndingBefore(overlappingInstanceStart);

                if (freeRepetitionsInFuture == 0) {
                    resourceAvailabilities.add(SubjectAvailability.known(email, Availability.BUSY));
                } else {
                    LocalDate due = EventRoutines.convertDueTsToUntilDate(overlappingInstanceStart, tz);
                    resourceAvailabilities.add(SubjectAvailability.availableDue(email, due, freeRepetitionsInFuture));
                }
            }
        }

        futureRepetitions = RepetitionInstanceSet.boundedForUsersAvailability(repetitionInfo, getActionInfo().getNow());
        request = AvailabilityRequest.intervals(futureRepetitions)
                .excludeEventId(params.exceptEventId)
                .excludeAbsencesEvents();

        ListF<AvailabilityOverlapOrRefusal> usersOverlaps = availRoutines.findFirstAvailabilityOverlaps(
                uid, userIds, request, getActionInfo());

        ListF<SubjectAvailability> usersAvailabilities = Cf.arrayList();
        for (AvailabilityOverlapOrRefusal overlap : usersOverlaps) {
            Email email = emailBySubjectId.getOrThrow(overlap.getSubjectId());

            if (!overlap.isRefusal()) {
                Option<Availability> avail = overlap.getOverlap().map(AvailabilityOverlap::getAvailability);
                usersAvailabilities.add(SubjectAvailability.known(email, avail.getOrElse(Availability.AVAILABLE)));
            } else {
                usersAvailabilities.add(SubjectAvailability.unknown(email));
            }
        }
        MapF<Email, SubjectAvailability> resources = resourceAvailabilities
                .toMapMappingToKey(SubjectAvailability::getEmail);

        MapF<Email, SubjectAvailability> users = usersAvailabilities
                .toMapMappingToKey(SubjectAvailability::getEmail);

        return emails.map(e -> resources.getO(e).orElse(() -> users.getO(e))
                .getOrElse(() -> SubjectAvailability.unknown(e)));
    }

    private <R> ListF<R> getAvailabilityResponses(
            PassportUid uid, ListF<Email> emails, AvailabilityRequest req,
            Function2<Email, AvailabilityIntervals, R> consSuccessResponseF,
            Function2<Email, AvailabilityQueryRefusalReason, R> consRefusalResponseF)
    {
        Tuple2<Tuple2List<Email, AvailabilityIntervalsOrRefusal>, Tuple2List<Email, AvailabilityIntervalsOrRefusal>> p =
                availRoutines.getAvailabilityIntervalssByEmails(uid, emails, req, getActionInfo())
                        .partitionBy2(AvailabilityIntervalsOrRefusal.isRefusalF().notF());

        Tuple2List<Email, R> success = p._1.map2(AvailabilityIntervalsOrRefusal.getIntervalsF())
                .zip3With(consSuccessResponseF).get13();
        Tuple2List<Email, R> refusal = p._2.map2(AvailabilityIntervalsOrRefusal.getRefusalReasonF())
                .zip3With(consRefusalResponseF).get13();

        return emails.map(success.plus(refusal).toMap()::getOrThrow);
    }

    private SuggestInfo.WithPlaces toSuggestInfoWithPlaces(
            UserInfo userInfo, SuggestData suggestRequest, Suggest suggest,
            MapF<Long, Comparator<ResourceInfo>> comparatorByOfficeId, DateTimeZone tz, Language lang)
    {
        Validate.isFalse(suggestRequest.isSameRoomMode());

        Option<InstantInterval> preferredInterval = findSuggestedIntervalWithClosestStart(suggestRequest, suggest, tz);

        MapF<Long, Integer> numberOfResourcesByOfficeId = suggestRequest.getNumbersOfResources();

        ListF<SuggestInfo.IntervalAndPlaces> intervals = Cf.arrayList();

        for (AvailableResourcesInOffices suggestForInterval : suggest.getSuggestedIntervals()) {
            ListF<SuggestInfo.Place> places = Cf.arrayList();

            ListF<OfficeAndFreeResources> officesAndResourcesInOrder = suggestRequest.getOfficeIds()
                    .filterMap(suggestForInterval.getOfficeAndFreeResources()
                            .toMapMappingToKey(OfficeAndFreeResources.getOfficeIdF())::getO);

            for (OfficeAndFreeResources officeAndResources : officesAndResourcesInOrder) {
                Office office = officeAndResources.getOffice();
                DateTimeZone officeTz = OfficeManager.getOfficeTimeZone(office);

                SuggestInfo.OfficeInfo officeInfo = new SuggestInfo.OfficeInfo(
                        office.getId(), ResourceRoutines.getNameI18n(office, lang));

                int numberOfOptions = suggestRequest.getNumberOfOptions().getOrElse(MAX_RESOURCE_OPTIONS_PER_OFFICE);
                int size = numberOfOptions >= 0
                        ? numberOfOptions * numberOfResourcesByOfficeId.getOrThrow(office.getId())
                        : Integer.MAX_VALUE;

                Tuple2List<ResourceInfo, Option<LocalDate>> availableResourcesSorted = officeAndResources
                        .getAvailableResources().takeSortedBy1(comparatorByOfficeId.getOrThrow(office.getId()), size);

                ListF<SuggestInfo.WebResourceInfo> suggestedResources =
                        availableResourcesSorted.map(toWebResourceInfoF(userInfo, lang));

                LocalDateTimeInterval officeLocalInterval = new LocalDateTimeInterval(
                        suggestForInterval.getInterval(), officeTz);

                boolean hasMore = officeAndResources.getAvailableResources().size() > size;

                places.add(new SuggestInfo.Place(officeInfo, officeLocalInterval, suggestedResources, hasMore));
            }

            InstantInterval interval = suggestForInterval.getInterval();
            LocalDateTimeInterval localInterval = new LocalDateTimeInterval(interval, tz);

            intervals.add(new SuggestInfo.IntervalAndPlaces(
                    localInterval, places, interval.equals(preferredInterval.get())));
        }

        return new SuggestInfo.WithPlaces(intervals, suggest.getBackSearchStart(tz), suggest.getNextSearchStart(tz));
    }

    private SuggestInfo.WithNoPlaces toSuggestInfoWithNoPlaces(
            PassportUid uid, SuggestData suggestRequest, Suggest suggest, DateTimeZone tz)
    {
        Validate.isTrue(suggestRequest.isSameRoomMode());

        Option<InstantInterval> preferred = findSuggestedIntervalWithClosestStart(suggestRequest, suggest, tz);

        ListF<Long> selectedResourceIds = findSelectedResources(uid, suggestRequest).map(ResourceInfo.resourceIdF());
        Function1B<ResourceInfo> isSelectedF = ResourceInfo.resourceIdF().andThen(selectedResourceIds.containsF());

        Comparator<LocalDate> nullHiComparator = Comparator.<LocalDate>constEqualComparator()
                .nullLowC().reversed().thenComparing(Comparator.naturalComparator());

        Comparator<Option<LocalDate>> dueDateComparator = nullHiComparator.compose(Option::getOrNull);

        ListF<SuggestInfo.IntervalAndOptions> intervals = Cf.arrayList();

        for (ListF<AvailableResourcesInOffices> options
                : suggest.getSuggestedIntervals().paginate(suggestRequest.getNumberOfOptions().get()))
        {
            ListF<InstantInterval> instantIntervals = options.map(AvailableResourcesInOffices::getInterval);

            ListF<SuggestInfo.IntervalWithDue> dueIntervals = options.map(o -> {
                Option<LocalDate> dueDate = Option.empty();

                for (OfficeAndFreeResources office : o.getOfficeAndFreeResources()) {
                    int numberOfResources = suggestRequest.getNumbersOfResources().getOrThrow(office.getOffice().getId());

                    ListF<Option<LocalDate>> selectedDues =
                            office.getAvailableResources().filterBy1(isSelectedF).get2();

                    ListF<Option<LocalDate>> notSelectedDues = selectedDues.size() < numberOfResources
                            ? office.getAvailableResources().filterBy1(isSelectedF.notF()).get2()
                            : Cf.list();

                    dueDate = Option.of(dueDate)
                            .plus(selectedDues.minO(dueDateComparator))
                            .plus(notSelectedDues.maxO(dueDateComparator))
                            .min(dueDateComparator);
                }
                return new SuggestInfo.IntervalWithDue(new LocalDateTimeInterval(o.getInterval(), tz), dueDate);
            });

            intervals.add(new SuggestInfo.IntervalAndOptions(
                    dueIntervals, instantIntervals.containsTs(preferred.get())));
        }
        return new SuggestInfo.WithNoPlaces(intervals, suggest.getBackSearchStart(tz), suggest.getNextSearchStart(tz));
    }

    private MapF<Long, Comparator<ResourceInfo>> getComparatorsForRequestedOffices(
            final PassportUid uid, ListF<PassportUid> knownUsers, SuggestData suggestRequest)
    {
        ListF<ResourceInfo> selectedResources = findSelectedResources(uid, suggestRequest);

        MapF<Long, ListF<ResourceInfo>> foundByOffice = selectedResources.groupBy(ResourceInfo.officeIdF());

        final MapF<Long, ListF<ResourceInfo>> selectedResourcesByOffice = suggestRequest.getOfficeIds()
                .toMapMappingToValue(id -> foundByOffice.getOrElse(id, Cf.list()));

        final MapF<Long, OfficeUsers> usersByOffice =
                groupUsersByOffices(suggestRequest.getOfficeIds(), knownUsers.plus(uid).stableUnique())
                        .toMapMappingToKey(OfficeUsers.getOfficeIdF());

        final Comparator<ResourceInfo> hasAnyPhoneForMultiOfficeComparator = suggestRequest.getOfficeIds().size() > 1
                ? resourceComparators.hasAnyPhoneComparator()
                : Comparator.constEqualComparator();

        final Comparator<ResourceInfo> floorComparator;

        if (suggestRequest.getNumberOfOptions().exists(Cf.Integer.ltF(0))) {
            floorComparator = ResourceInfo.resourceF().andThen(
                    ResourceFields.FLOOR_NUM.getF().andThenNaturalComparator().thenComparing(
                            ResourceComparators.BY_GROUP_NAME.uncheckedCastC()));
        } else {
            floorComparator = Comparator.constEqualComparator();
        }

        return suggestRequest.getOfficeIds().toMapMappingToValue(officeId -> {
            ListF<ResourceInfo> selectedResources1 = selectedResourcesByOffice.getOrThrow(officeId);
            OfficeUsers users = usersByOffice.getOrThrow(officeId);

            OfficeFloor mostUsersFloor = users.findMostUsersFloor(uid);
            int usersCount = users.getUsersCount();

            Comparator<ResourceInfo> distanceComparator = mostUsersFloor.getFloorNum().isPresent()
                    ? resourceComparators.distanceFromFloorComparator(mostUsersFloor)
                    : resourceComparators.distanceFromSelectedResourcesComparator(selectedResources1);

            return floorComparator.thenComparing(
                    resourceComparators.hasIdComparator(selectedResources1.map(ResourceInfo.resourceIdF()))).thenComparing(
                    resourceComparators.hasEnoughCapacityComparator(usersCount)).thenComparing(
                    hasAnyPhoneForMultiOfficeComparator).thenComparing(
                    resourceComparators.hasVideoConferencingComparator().reversed()).thenComparing(
                    distanceComparator);
        });
    }

    private ListF<ResourceInfo> findSelectedResources(PassportUid uid, SuggestData suggestRequest) {
        ListF<Email> selected = suggestRequest.getOffices().flatMap(SuggestData.Office.getSelectedResourceEmailsF());

        return resourceRoutines.getDomainResourcesCanViewWithLayersAndOffices(
                uid, OfficeFilter.any(), ResourceFilter.any().withEmailFilter(selected));
    }

    private ListF<OfficeUsers> groupUsersByOffices(ListF<Long> officeIds, ListF<PassportUid> knownUsers) {
        ListF<SettingsYt> settings = settingsRoutines.getSettingsByUidBatch(knownUsers).values()
                .filterMap(SettingsInfo.getYtF())
                .filter(s -> s.getActiveOfficeId().isPresent());

        Function<SettingsYt, Long> officeIdF = SettingsYt.getActiveOfficeIdF().andThen(Cf2.f(Option::get));

        MapF<Long, ListF<OfficeUsers.User>> usersByOfficeId = Cf.hashMap();

        ListF<SettingsYt> requested = settings.filter(officeIdF.andThen(officeIds.containsF()));
        ListF<SettingsYt> notRequested = settings.filter(officeIdF.andThen(officeIds.containsF().notF()));

        ListF<Office> offices = officeManager.getOfficesByIds(settings.map(officeIdF).plus(officeIds));

        Function<Long, Option<String>> cityNameF = offices.toMap(Office.getIdF(), Office.getCityNameF())::getOrThrow;
        Function<Long, Option<Long>> findByCityInRequestedF = cityNameF.andThen(
                Cf2.f(requested.map(officeIdF).toMapMappingToKey(cityNameF)::getO));

        for (SettingsYt s : requested) {
            Option<Integer> floorNum =
                    s.getTableOfficeId().equals(s.getActiveOfficeId()) ? s.getTableFloorNum() : Option.empty();

            OfficeUsers.User user = new OfficeUsers.User(s.getUid(), floorNum);
            usersByOfficeId.getOrElseUpdate(s.getActiveOfficeId().get(), Cf::arrayList).add(user);
        }
        for (Tuple2<SettingsYt, Long> t : notRequested.zipWithFlatMapO(officeIdF.andThen(findByCityInRequestedF))) {
            OfficeUsers.User user = new OfficeUsers.User(t._1.getUid(), Option.empty());
            usersByOfficeId.getOrElseUpdate(t._2, Cf::arrayList).add(user);
        }

        return officeIds.zipWith(id -> usersByOfficeId.getOrElse(id, Cf.list())).map(OfficeUsers.consF());
    }

    private static Function2<ResourceInfo, Option<LocalDate>, SuggestInfo.WebResourceInfo> toWebResourceInfoF(
            UserInfo userInfo, Language lang) {
        return (resourceInfo, dueDate) -> toWebResourceInfo(userInfo, resourceInfo, dueDate, lang);
    }

    private static SuggestInfo.WebResourceInfo toWebResourceInfo(
            UserInfo userInfo, ResourceInfo resourceInfo, Option<LocalDate> dueDate, Language lang)
    {
        return new SuggestInfo.WebResourceInfo(
                resourceInfo.getResourceId(), resourceInfo.getNameI18n(lang).getOrElse(""),
                resourceInfo.getEmail(), resourceInfo.getResource().getType(), dueDate,
                resourceInfo.getPhone().isPresent(), resourceInfo.getVideo().isPresent(),
                resourceInfo.getFloorNum(), resourceInfo.getGroupNameI18n(lang),
                Option.when(userInfo.canAdminResource(resourceInfo.getResource()), true));
    }

    private static Option<InstantInterval> findSuggestedIntervalWithClosestStart(
            SuggestData suggestRequest, Suggest suggest, DateTimeZone tz)
    {
        if (suggest.getSuggestedIntervals().isEmpty()) {
            return Option.empty();
        }
        return Option.of(findIntervalWithClosestStart(
                AuxDateTime.toInstantIgnoreGap(suggestRequest.getEventStart(), tz).toInstant(),
                suggest.getSuggestedIntervals().map(AvailableResourcesInOffices.getIntervalF())));
    }

    private static InstantInterval findIntervalWithClosestStart(final Instant from, ListF<InstantInterval> intervals) {
        return intervals.min(AuxDateTime.millisToF(from).compose(InstantInterval::getStart).andThenNaturalComparator());
    }

    private SearchInterval getSearchInterval(SuggestData data, DateTimeZone tz) {
        Validate.isTrue(!data.getSearchBackward().isPresent() || data.getSearchStart().isPresent());

        Duration gap = new Duration(getActionInfo().getNow(), data.getEventStart(tz));

        Instant since = gap.isShorterThan(Duration.ZERO) && !gap.isShorterThan(Duration.standardMinutes(-30))
                ? data.getEventStart(tz) : getActionInfo().getNow();

        if (!data.getSearchStart().isPresent()) {
            Instant startOfEventDay = data.getEventStart().toLocalDate().toDateTimeAtStartOfDay(tz).toInstant();
            Instant start = ObjectUtils.max(startOfEventDay, since);

            return SearchInterval.forward(new InstantInterval(start, start.plus(SUGGEST_SEARCH_DURATION)));
        }
        Instant start = data.getSearchStart().get().toDateTime(tz).toInstant();

        if (!data.getSearchBackward().isSome(true)) {
            Instant max = ObjectUtils.max(start, since);
            return SearchInterval.forward(new InstantInterval(max, max.plus(SUGGEST_SEARCH_DURATION)));

        } else {
            Instant max = ObjectUtils.max(start.minus(SUGGEST_SEARCH_DURATION), since);
            return SearchInterval.backward(
                    max.isBefore(start) ? new InstantInterval(max, start) : new InstantInterval(max, max));
        }
    }

    private ActionInfo getActionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }
}
