package ru.yandex.calendar.frontend.webNew;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.bender.WebDate;
import ru.yandex.calendar.frontend.web.cmd.run.ResourceBusyOverlapException;
import ru.yandex.calendar.frontend.webNew.dto.in.RepetitionWithStartData;
import ru.yandex.calendar.frontend.webNew.dto.in.ReplyData;
import ru.yandex.calendar.frontend.webNew.dto.in.WebEventData;
import ru.yandex.calendar.frontend.webNew.dto.in.WebEventUserData;
import ru.yandex.calendar.frontend.webNew.dto.inOut.EventAttachmentData;
import ru.yandex.calendar.frontend.webNew.dto.out.DeletedEventInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.EventBrief;
import ru.yandex.calendar.frontend.webNew.dto.out.EventInfoShort;
import ru.yandex.calendar.frontend.webNew.dto.out.EventInfoWithSeries;
import ru.yandex.calendar.frontend.webNew.dto.out.EventsBrief;
import ru.yandex.calendar.frontend.webNew.dto.out.EventsCountInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.EventsInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ExdateBrief;
import ru.yandex.calendar.frontend.webNew.dto.out.ModifiedEventIdsInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ModifyEventResult;
import ru.yandex.calendar.frontend.webNew.dto.out.MoveResourceEventsIds;
import ru.yandex.calendar.frontend.webNew.dto.out.ParseTimeInEventNameResult;
import ru.yandex.calendar.frontend.webNew.dto.out.RepetitionDescription;
import ru.yandex.calendar.frontend.webNew.dto.out.ResourcesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.StatusResult;
import ru.yandex.calendar.frontend.webNew.dto.out.WebEventInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.WebResourceInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.WebUserInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.WebUserParticipantInfo;
import ru.yandex.calendar.logic.beans.Bean;
import ru.yandex.calendar.logic.beans.generated.DeletedEvent;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventAttachment;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.beans.generated.EventUserFields;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.MainEventFields;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.CreateInfo;
import ru.yandex.calendar.logic.event.EventActions;
import ru.yandex.calendar.logic.event.EventDbManager;
import ru.yandex.calendar.logic.event.EventGetProps;
import ru.yandex.calendar.logic.event.EventInfoDbLoader;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventLayerWithRelations;
import ru.yandex.calendar.logic.event.EventLoadLimits;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventUserWithRelations;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.event.EventsFilter;
import ru.yandex.calendar.logic.event.LayerIdPredicate;
import ru.yandex.calendar.logic.event.WebNewEventDataConverter;
import ru.yandex.calendar.logic.event.archive.DeletedEventDao;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.event.model.EventData;
import ru.yandex.calendar.logic.event.model.EventInvitationData;
import ru.yandex.calendar.logic.event.model.EventInvitationsData;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.repetition.EventIndentInterval;
import ru.yandex.calendar.logic.event.repetition.EventIndentIntervalAndPerms;
import ru.yandex.calendar.logic.event.repetition.RepetitionConfirmationManager;
import ru.yandex.calendar.logic.event.repetition.RepetitionInstanceInfo;
import ru.yandex.calendar.logic.event.repetition.RepetitionRoutines;
import ru.yandex.calendar.logic.event.repetition.RepetitionToStringConverter;
import ru.yandex.calendar.logic.event.repetition.RepetitionUtils;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.event.web.IdOrExternalId;
import ru.yandex.calendar.logic.event.web.UserOrYaCalendar;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.layer.LayerUserWithRelations;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sending.so.SoChecker;
import ru.yandex.calendar.logic.sharing.InvitationProcessingMode;
import ru.yandex.calendar.logic.sharing.participant.EventParticipants;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.ResourceParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.EventActionClass;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.dates.HumanReadableTimeParser;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.time.TimeUtils;

import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.function.Function.identity;
import static org.joda.time.DateTimeZone.UTC;

@Slf4j
public class WebNewEventManager {
    static final int ATTENDEE_LIMIT = 10;

    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private UserManager userManager;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private EventDbManager eventDbManager;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private RepetitionConfirmationManager repetitionConfirmationManager;
    @Autowired
    private WebNewLayerManager webNewLayerManager;
    @Autowired
    private WebNewUserManager webNewUserManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private EventInfoDbLoader eventInfoDbLoader;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private DeletedEventDao deletedEventDao;
    @Autowired
    private WebNewResourcesManager webNewResourcesManager;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private RepetitionRoutines repetitionRoutines;
    @Autowired
    private SoChecker soChecker;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    public ModifyEventResult createEvent(PassportUid uid, Option<String> extId, Option<DateTimeZone> tzO,
                                         WebEventData createEventData, ActionInfo actionInfo) {
        val tz = tzO.getOrElse(() -> dateTimeManager.getTimeZoneForUid(uid));
        try {
            createEventData.getLayerId()
                .ifPresent(layerId -> {
                    val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layerRoutines.getLayerById(layerId));
                    authorizer.ensureLayerType(layerPermInfo, LayerType.USER);
                });

            EventData eventData = WebNewEventDataConverter.convert(createEventData, tz, tz);
            val notifications = createEventData.getNotifications()
                    .map(NotificationsData::createFromWeb)
                    .getOrElse(NotificationsData::useLayerDefaultIfCreate);

            eventData.setExternalId(extId);

            val type = eventData.getEvent().getFieldValueO(EventFields.TYPE);
            val mode = type.exists(EventType::isAbsence)
                    ? InvitationProcessingMode.SAVE_ATTACH
                    : InvitationProcessingMode.SAVE_ATTACH_SEND;

            final EventUser eu = eventData.getEventUserData().getEventUser().copy();
            eu.setFieldValueDefault(EventUserFields.AVAILABILITY,
                    type.isSome(EventType.TRIP) ? Availability.AVAILABLE : Availability.BUSY); // CAL-4598

            eventData.setEventUserData(eventData.getEventUserData().withEventUser(eu));

            soChecker.validateEvent(uid, eventData, actionInfo);

            CreateInfo createInfo = eventWebManager.createUserEvent(
                    uid, eventData, notifications, mode, actionInfo);

            val closest = createInfo.getEventAndRepetition().getClosestInterval(Instant.now()).get();

            return ModifyEventResult.modified(
                    closest.getEventId(), new LocalDate(closest.getStart(), tz),
                    closest.getEvent().getSequence(), createInfo.getExternalId());
        } catch (ResourceBusyOverlapException e) {
            log.warn("ResourceBusyOverlapException occurred", e);
            return ModifyEventResult.resourceBusyOverlap(e.getResource(), e.getOverlap(), tz);
        }
    }

    public ModifyEventResult updateEvent(PassportUid uid, IdOrExternalId eventId, Option<Long> layerId,
                                         Option<String> privateToken, Option<Integer> sequence, Option<DateTimeZone> tzO,
                                         WebEventData webEventData, Option<Boolean> applyToFuture, Option<LocalDateTime> instanceStartTs,
                                         Option<Boolean> mailToAll, ActionInfo actionInfo) {
        val dataTz = tzO.getOrElse(dateTimeManager.getTimeZoneForUid(uid));
        val eventTz = eventWebManager.getEventTimezoneIfExists(uid, eventId).getOrElse(dataTz);
        try {
            if (applyToFuture.isSome(false)) { // to prevent moving start to day of repetition by AbstractEventDataConverter
                webEventData.setRepetitionNone();
            }
            val eventData = WebNewEventDataConverter.convert(webEventData, dataTz, eventTz);
            if (eventId.isExternal()) {
                eventData.setExternalId(Option.of(eventId.getExternalId()));
            } else {
                eventData.getEvent().setId(eventId.getId());
            }
            layerId.forEach(eventData::setPrevLayerId);

            soChecker.validateEvent(uid, eventData, actionInfo);

            val notifications = webEventData.getNotifications()
                    .map(NotificationsData::updateFromWeb)
                    .getOrElse(NotificationsData::notChanged);

            if (instanceStartTs.isPresent()) {
                eventData.setInstanceStartTs(instanceStartTs.get().toDateTime(DateTimeZone.UTC).toInstant());
            }

            val updateInfo = eventWebManager.update(
                    userManager.getUserInfo(uid), eventData, notifications,
                    privateToken, sequence, applyToFuture.getOrElse(false), mailToAll, actionInfo);

            val event = updateInfo.getNewEvent()
                    .getOrElse(updateInfo.getUpdatedEvent()::get);

            val closestO = event.getClosestInterval(actionInfo.getNow());

            if (closestO.isPresent()) {
                val closest = closestO.get();
                return ModifyEventResult.modified(
                        closest.getEventId(), new LocalDate(closest.getStart(), dataTz),
                        closest.getEvent().getSequence(), updateInfo.getExternalIds());
            } else {
                return ModifyEventResult.modified(event.getEvent().getId(),
                        new LocalDate(Instant.now(), dataTz),
                        event.getEvent().getSequence(), updateInfo.getExternalIds());
            }
        } catch (ResourceBusyOverlapException e) {
            log.warn("ResourceBusyOverlapException occurred", e);
            return ModifyEventResult.resourceBusyOverlap(e.getResource(), e.getOverlap(), dataTz);
        }
    }

    public StatusResult attachEvent(PassportUid uid, long eventId, WebEventUserData data, ActionInfo actionInfo) {
        EventUser eventUser = data.getEventUserData().getEventUser().copy();
        eventUser.setIsSubscriber(true);

        eventWebManager.attachEvent(
                userManager.getUserInfo(uid), eventId, data.getLayerId(),
                eventUser, data.getNotificationsCreateData(), actionInfo);

        return StatusResult.ok();
    }

    public StatusResult deleteEvent(PassportUid uid, IdOrExternalId eventId, Option<Integer> sequence, boolean applyToFuture,
                                    Option<LocalDateTime> instanceStartTs, ActionInfo actionInfo) {
        Option<Instant> instanceStart = instanceStartTs.map(TimeUtils.localDateTime.toInstantF(DateTimeZone.UTC));

        UserInfo user = userManager.getUserInfo(uid);
        eventWebManager.deleteEvent(user, eventId, sequence, instanceStart, applyToFuture, actionInfo);

        return StatusResult.ok();
    }

    public StatusResult detachEvent(PassportUid uid, long eventId, Option<Long> layerId, ActionInfo actionInfo) {
        eventWebManager.detachEvent(userManager.getUserInfo(uid), eventId, layerId, actionInfo);
        return StatusResult.ok();
    }

    public StatusResult confirmEventRepetition(long eventId, ActionInfo actionInfo) {
        repetitionConfirmationManager.confirm(eventDao.findMainEventIdByEventId(eventId).get(), actionInfo);
        return StatusResult.ok();
    }

    public StatusResult deleteFutureEvents(PassportUid uid, long eventId, ActionInfo actionInfo) {
        eventWebManager.deleteFutureEvents(UserOrYaCalendar.user(userManager.getUserInfo(uid)), eventId, actionInfo);
        return StatusResult.ok();
    }

    private boolean showCreatorAsOrganizer(Participants participants,
                                           Option<Layer> layer, Option<PassportUid> targetUid) {
        return passportAuthDomainsHolder.containsYandexTeamRu()
                && targetUid.isPresent()
                && layer.isMatch(l -> !l.getCreatorUid().equalsTs(targetUid.get()))
                && participants.isNotMeetingStrict();
    }

    private Option<WebUserParticipantInfo> getCreatorFromEvent(EventParticipants eventParticipants, Event event) {
        Option<EventUserWithRelations> creator = eventParticipants.getEventUser(event.getCreatorUid());
        return creator.map(u -> new WebUserParticipantInfo(WebUserInfo.fromUserParticipantInfo(
                new YandexUserParticipantInfo(u.getEventUser(), u.getSettings())), u.getDecision())
        );
    }

    public EventInfoWithSeries getEvent(PassportUid uid, String externalId, Language lang, ActionInfo actionInfo) {
        val masterEvent = eventDao.findEventsByExternalIdAndRecurrenceId(externalId, Option.empty()).first();
        val id = IdOrExternalId.externalId(externalId);
        val event = getEvent(uid, id, lang, Optional.of(masterEvent.getStartTs()), actionInfo);

        val repetition = repetitionRoutines.getRepetitionInstanceInfoByEventId(event.getId());
        val exdates = repetition.getExdates().map(ExdateBrief::new);
        val recurrentEvents = StreamEx.of(repetition.getRecurIds())
                .nonNull()
                .map(start -> getEvent(uid, id, lang, Optional.of(start), actionInfo))
                .toImmutableList();

        return new EventInfoWithSeries(event, recurrentEvents, exdates);
    }

    public WebEventInfo getEvent(PassportUid uid, IdOrExternalId eventId, Language lang, Optional<Instant> instanceStartTs,
                                 ActionInfo actionInfo) {
        return getEvent(uid, eventId,
                Option.empty(), Option.empty(), Option.empty(),
                Option.x(instanceStartTs).map(LocalDateTime::new), Option.empty(), Option.empty(),
                lang, Optional.empty(), actionInfo);
    }

    public WebEventInfo getEvent(PassportUid uid, IdOrExternalId eventId, Option<Long> layerId,
                                 Option<String> privateToken, Option<DateTimeZone> tzO,
                                 Option<LocalDateTime> instanceStartTs, Option<Boolean> recurrenceAsOccurrence,
                                 Option<String> linkSignKey, Language lang, Optional<PassportUid> participantUid,
                                 ActionInfo actionInfo) {
        val targetUid = participantUid.orElse(uid);

        val tz = tzO.getOrElse(dateTimeManager.getTimeZoneForUid(targetUid));
        val instanceStart = instanceStartTs.map(TimeUtils.localDateTime.toInstantF(DateTimeZone.UTC));

        val user = userManager.getUserInfo(targetUid);

        val e = eventWebManager.getModificationEvent(Option.of(targetUid), eventId, instanceStart);

        val instance = recurrenceAsOccurrence.isSome(true)
                ? eventRoutines.getRecurrenceInstanceAsOccurrence(Option.of(targetUid), e.getId(), actionInfo.getActionSource())
                : eventRoutines.getSingleInstance(Option.of(targetUid), instanceStart, true, false, e.getId(), actionInfo.getActionSource());

        val event = instance.getEventWithRelations();

        val layer = eventRoutines.findUserLayerWithEvent(targetUid, layerId, event)
                .map(EventLayerWithRelations::getLayer);

        layerId = layer.map(Layer::getId);
        EventActions actions = eventRoutines.getEventActions(user, event, layer, actionInfo.getActionSource());

        val knowsToken = privateToken.isPresent()
                && eventWebManager.findEventInvitation(event, privateToken.get()).isPresent();


        participantUid.ifPresentOrElse(
            p -> {
                val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, event);
                authorizer.ensureCanViewEventForParticipant(user, uid, eventAuthInfo, knowsToken, actionInfo.getActionSource());
            },
            () -> {
                val eventAuthInfo = authorizer.loadEventInfoForPermsCheck(user, event);
                authorizer.ensureCanViewEvent(user, eventAuthInfo, knowsToken, actionInfo.getActionSource());
            }
        );

        if (knowsToken && event.getParticipants().isMeeting() && !event.getParticipants().isParticipantWithInconsistent(targetUid)) {
            actions = actions.withAccept(true).withReject(true);
        }

        Option<WebUserParticipantInfo> organizer = Option.empty();
        ListF<WebResourceInfo> resources = Cf.list();
        ListF<WebUserParticipantInfo> attendees = Cf.list();
        ListF<WebUserParticipantInfo> optionalAttendees = Cf.list();
        val participants = event.getParticipants();

        if (participants.isMeeting()) {
            Option<Office> userOffice = participants.getResourceParticipants().size() > 1
                    ? officeManager.getDefaultOfficeForUser(targetUid)
                    : Option.empty();

            val dt = instance.getInterval().getStart().toDateTime(tz);

            resources = webNewResourcesManager.sortedByOffices(participants.getResourceParticipants()
                    .map(ResourceParticipantInfo::getResourceInfo), userOffice, lang)
                    .map(r -> webNewResourcesManager.toWebResourceInfo(Option.of(user), r, dt, lang));

            attendees = toWebUserParticipantInfos(targetUid, participants.getUserAttendeesButNotOrganizer(), lang);
            optionalAttendees = toWebUserParticipantInfos(targetUid, participants.getUserOptionalAttendeesButNotOrganizer(), lang);
            organizer = toWebUserParticipantInfos(
                    targetUid, participants.getOrganizerSafe().filterByType(UserParticipantInfo.class), lang).singleO();
        } else if (showCreatorAsOrganizer(participants, layer, Option.of(targetUid))) {
            organizer = getCreatorFromEvent(event.getEventParticipants(), event.getEvent());
        }
        val subscribers = toWebUserInfos(targetUid, participants.getNotDeclinedSubscribers(), lang);

        val decision = instance.getEventUser().map(EventUser.getDecisionF());
        val availability = instance.getEventUser().map(EventUser.getAvailabilityF());

        val repetitionNeedsConfirmation = participants.isOrganizer(targetUid)
                && repetitionConfirmationManager.isItTimeToConfirm(event.getMainEventId(), actionInfo);

        if (instance.getInfoForPermsCheck().isOrganizerLetToEditAnyMeeting() && participants.isAttendee(targetUid)) {
            actions = actions.withEdit(true);
        }

        val canAdminAllResources = actions.canEdit();

        val attachments = toWebAttachmentInfos(instance.getAttachmentsO());

        return WebEventInfo.create(
                instance.getEventInterval(),
                instance.getMainEvent().getExternalId(),
                instance.getEvent(),
                attachments,
                instance.getInfoForPermsCheck(),
                instance.getRepetition(),
                repetitionNeedsConfirmation,
                organizer,
                resources,
                attendees,
                optionalAttendees,
                subscribers,
                instance.getNotifications(NotificationsData.WEB_CHANNELS),
                actions,
                decision,
                availability,
                layerId,
                tz,
                linkSignKey,
                canAdminAllResources);
    }

    private Option<ListF<EventAttachmentData>> toWebAttachmentInfos(Option<ListF<EventAttachment>> attachments) {
        return attachments.map(list -> list.map(EventAttachmentData::fromEventAttachment));
    }

    public EventsInfo getEvents(
            final Optional<PassportUid> actorUid, Optional<PassportUid> targetUid,
            ListF<String> layerStrIds, Option<String> layerToken, Option<Boolean> layerToggledOn,
            WebDate from, Option<WebDate> till,
            Option<Boolean> mergeLayers, Option<Boolean> opaqueOnly,
            Option<Boolean> showDeclined, Option<Boolean> hideAbsences,
            Optional<DateTimeZone> tzO, Option<Language> langO, Option<String> linkSignKey, boolean limitAttendees,
            ActionInfo actionInfo, Option<Integer> limitSize) {
        return getEvents(actorUid, targetUid, layerStrIds, layerToken, layerToggledOn, from, till, mergeLayers,
                opaqueOnly, showDeclined, hideAbsences, tzO, langO, linkSignKey, Option.empty(), Option.empty(),
                limitAttendees, actionInfo, limitSize);
    }

    public EventsInfo getEvents(
            final Optional<PassportUid> actorUid, Optional<PassportUid> targetUid,
            ListF<String> layerStrIds, Option<String> layerToken, Option<Boolean> layerToggledOn,
            WebDate from, Option<WebDate> till,
            Option<Boolean> mergeLayers, Option<Boolean> opaqueOnly,
            Option<Boolean> showDeclined, Option<Boolean> hideAbsences,
            Optional<DateTimeZone> tz, Option<Language> langO, Option<String> linkSignKey,
            Option<Long> eventIdO, ListF<String> externalIds, boolean limitAttendees,
            ActionInfo actionInfo, Option<Integer> limitSize) {
        targetUid = targetUid.or(() -> actorUid);
        val foundAndNot = webNewLayerManager.findLayersCanList(
                Option.x(targetUid), layerStrIds, layerToken, layerToggledOn, !hideAbsences.isSome(true), opaqueOnly.isSome(true));

        val layers = foundAndNot.get1();

        val tokenAccessedLayerId = layers
                .find(Layer.getPrivateTokenF().andThen(Cf2.isSomeOfF(layerToken))).map(Layer.getIdF());
        var finalActorUid = actorUid;
        if (actorUid.isEmpty()) {
            val layerUsersWithRelationsByLayerId = layerRoutines.getLayerUsersWithRelationsByLayerIds(tokenAccessedLayerId).firstO();
            if (layerUsersWithRelationsByLayerId.isPresent()) {
                finalActorUid = Optional.of(layerUsersWithRelationsByLayerId.get().getLayerUser().getUid());
            }
        }
        val failedLayers = Option.when(layerStrIds.isNotEmpty() || layerToken.isPresent(), foundAndNot.get2());

        if (layers.isEmpty()) {
            return new EventsInfo(Cf.list(), Option.empty(), Option.empty(), failedLayers);
        }

        Optional<PassportUid> finalActorUid1 = finalActorUid;
        Supplier<Optional<DateTimeZone>> getUserTz = () -> finalActorUid1.map(dateTimeManager::getTimeZoneForUid);
        val tzValue = tz.or(getUserTz).orElseThrow(() -> new IllegalArgumentException("Uid or tz required"));

        val actorUserInfo = userManager.getUserInfos(Option.x(finalActorUid1)).singleO();
        val lang = langO.isPresent() ? langO.get() : settingsRoutines.getLanguageOrGetDefault(Option.x(finalActorUid1));

        val filter = new EventsFilter(
                tokenAccessedLayerId, showDeclined.isSome(true), opaqueOnly.isSome(true), mergeLayers.isSome(true));

        val egp = EventGetProps.none()
                .loadEventAllFields()
                .loadEventUserWithNotifications()
                .loadMainEventField(Cf.list(MainEventFields.EXTERNAL_ID, MainEventFields.LAST_UPDATE_TS))
                .loadEventParticipants()
                .loadEventAttachments()
                .excludeSubscribers();

        EventLoadLimits limits = EventLoadLimits.intersectsIntervalOptionalEnd(from.toInstant(tzValue), till.isPresent() ? Option.of(till.get().toInstantNextDay(tzValue)) : Option.empty());
        if (limitSize.isPresent()) {
            limits = limits.withResultSize(limitSize.get());
        }

        if (eventIdO.isPresent()) {
            limits = limits.withHaveId(eventRoutines.findMasterAndSingleEventIds(eventIdO.get()));
        } else if (externalIds.isNotEmpty()) {
            limits = limits.withHaveId(eventRoutines.findEventIdsByLayerIdsAndExternalIds(
                    layers.map(Layer::getId), externalIds));
        }

        val events = eventRoutines.getSortedInstancesIMayView(
                actorUserInfo, egp, layers, limits, filter, actionInfo.getActionSource());

        val infos = toWebEventInfos(actorUserInfo, events, layers, linkSignKey, tzValue, lang, actionInfo);

        val lastUpdateTs = layers.iterator().map(Layer::getCollLastUpdateTs)
                .plus(events.iterator().map(e -> e.getMainEvent().getLastUpdateTs())).maxO();

        return new EventsInfo(limitAttendees(infos, limitAttendees), Option.empty(), lastUpdateTs, failedLayers);
    }

    private ListF<WebEventInfo> limitAttendees(ListF<WebEventInfo> infos, boolean limitAttendees) {
        if (limitAttendees) {
            return infos.map(info -> info
                    .withAttendees(info.getAttendees().take(ATTENDEE_LIMIT))
                    .withOptionalAttendees(info.getOptionalAttendees().take(ATTENDEE_LIMIT))
            );
        }
        return infos;
    }

    public EventsInfo getModifiedEvents(
            PassportUid actorUid, Optional<PassportUid> targetUid, ListF<String> layerStrIds, Option<Boolean> layerToggledOn,
            Option<Instant> since, Option<DateTimeZone> tzO, Option<Language> langO, Option<String> linkSignKey, ActionInfo actionInfo) {
        val targetUidValue = targetUid.orElse(actorUid);
        val tz = tzO.getOrElse(dateTimeManager.getTimeZoneForUidF().bind(actorUid));
        val lang = langO.getOrElse(() -> settingsRoutines.getLanguage(actorUid));

        val actorUserInfo = userManager.getUserInfo(actorUid);

        val foundAndNot = webNewLayerManager.findLayersCanList(
                Option.of(targetUidValue), layerStrIds, Option.empty(), layerToggledOn, false, false);

        val layers = foundAndNot.get1();
        val failedLayers = foundAndNot.get2();

        val layerIds = layers.map(Layer::getId);

        val layerIdsMainEventIds = mainEventDao.findMainEventIdsOnLayersUpdatedAfter(layerIds, since.getOrElse(new Instant(0)));
        val deletedEvents = deletedEventDao.findEventsDeletedAfterByLayerIds(layerIds, since.getOrElse(new Instant(0)));

        val egp = EventGetProps.none()
                .loadEventAllFields().loadEventUserWithNotifications().loadEventParticipants()
                .loadMainEventField(Cf.list(MainEventFields.EXTERNAL_ID, MainEventFields.LAST_UPDATE_TS));

        val events = eventInfoDbLoader.getEventInfosByMainEventIds(
                Option.of(actorUserInfo), layerIdsMainEventIds.get2(), egp, actionInfo.getActionSource());

        val layerIdsByMainId = layerIdsMainEventIds.invert().groupBy1();
        SetF<Tuple2<Long, String>> updatedIds = Cf.hashSet();

        final var instances = events
                .flatMap(event -> layerIdsByMainId
                        .getOrThrow(event.getMainEventId())
                        .filterMap(layerId -> {
                            updatedIds.add(Tuple2.tuple(layerId, event.getMainEvent().getExternalId()));
                            if (!event.mayView()) {
                                return Option.empty();
                            }
                            val indent = new EventIndentInterval(
                                    event.getIndentAndRepetition(),
                                    event.getRepetitionInstanceInfo().getEventInterval());
                            return Option.of(EventRoutines.toEventInstanceInfoFromLayer(indent, event));
                        }));

        val eventInfos = toWebEventInfos(Option.of(actorUserInfo), instances, layers, linkSignKey, tz, lang, actionInfo);
        final var typeByLayerId = layers.toMap(Bean::getId, l -> l.getType().getMyEventType());

        final var deletedInfos = deletedEvents
                .filter(updatedIds.containsF().notF().compose(Tuple2.map2F(DeletedEvent::getExternalId)))
                .groupBy(t -> t.get2().getExternalId())
                .mapEntries((id, ts) -> {
                    val last = ts.maxBy(t -> t.get2().getDeletionTs());
                    return new DeletedEventInfo(
                            last.get1(),
                            typeByLayerId.getOrThrow(last.get1()),
                            last.get2().getExternalId(),
                            last.get2().getDeletionTs());
                });

        val lastUpdateTs = StreamEx.of(layers)
                .map(Layer::getCollLastUpdateTs)
                .append(deletedInfos.stream().map(DeletedEventInfo::getDeletionTs))
                .append(events.stream().map(e -> e.getMainEvent().getLastUpdateTs()))
                .max(Instant::compareTo);

        return new EventsInfo(eventInfos, Option.of(deletedInfos), Option.x(lastUpdateTs), Option.of(failedLayers));
    }

    public ModifiedEventIdsInfo getModifiedEventIds(PassportUid uid, Instant since, ListF<String> layerStrIds,
                                                    Option<Boolean> layerToggledOn) {
        MasterSlaveContextHolder.PolicyHandle handle = MasterSlaveContextHolder.push(MasterSlavePolicy.R_SM);
        try {
            Tuple2<ListF<Layer>, ListF<String>> foundAndNot = webNewLayerManager.findLayersCanList(
                    Option.of(uid), layerStrIds, Option.empty(), layerToggledOn, false, false);

            ListF<Layer> layers = foundAndNot.get1();
            Tuple2<ListF<Layer>, ListF<Layer>> feedAndNot = layers.partition(l -> l.getType() == LayerType.FEED);

            ListF<Long> layerIds = feedAndNot.get2().map(Layer::getId);

            ListF<Long> deletedEventLayerEventIds = deletedEventDao
                    .findEventIdsByEventLayerIdsDeletedAfter(layerIds, since);

            Tuple2List<String, Instant> extIdsAndTs = Tuple2List.arrayList();

            extIdsAndTs.addAll(deletedEventDao.findDeletedEventExternalIdsAndTiestampsByIds(deletedEventLayerEventIds));
            extIdsAndTs.addAll(mainEventDao.findExternalIdsAndTimestampsByEventIds(deletedEventLayerEventIds));
            extIdsAndTs.addAll(mainEventDao.findMainEventExternalIdsAndTimestampsOnLayersUpdatedAfter(layerIds, since));

            ListF<Long> icsFeedIds = feedAndNot.get1()
                    .filterMap(l -> Option.when(l.getCollLastUpdateTs().isAfter(since), l.getId()));

            Option<Instant> lastUpdateTs = layers.iterator().map(Layer::getCollLastUpdateTs)
                    .plus(extIdsAndTs.iterator().map(Tuple2::get2)).maxO();

            return new ModifiedEventIdsInfo(
                    extIdsAndTs.get1().stableUnique(), icsFeedIds, lastUpdateTs, Option.of(foundAndNot.get2()));
        } finally {
            handle.popSafely();
        }
    }

    public EventsBrief getEventsBrief(PassportUid uid, ListF<Long> eventIds, boolean forResource, Language lang,
                                      ActionInfo actionInfo) {
        val userInfo = userManager.getUserInfo(uid);

        ListF<EventWithRelations> events = eventDbManager.getEventsWithRelationsByIdsSafe(eventIds);
        authorizer.loadBatchAndCacheAllRequiredForPermsCheck(uid, events);

        MapF<ParticipantId, WebUserInfo> webUserInfoById = webNewUserManager.getWebUserInfos(
                uid, Either.right(events.flatMap(e -> e.getParticipants().getUserParticipantsSafe())), lang).toMap();

        val eventAuthInfoById = authorizer.loadEventsInfoForPermsCheck(userInfo, events);

        return new EventsBrief(events.map(event -> {
            val eventId = event.getId();
            val canView = authorizer.canViewEvent(userInfo, eventAuthInfoById.get(eventId), actionInfo.getActionSource());

            Function<ParticipantInfo, Option<WebUserParticipantInfo>> consUser = p -> Option.when(
                    p.getId().isAnyUser(),
                    () -> new WebUserParticipantInfo(webUserInfoById.getOrThrow(p.getId()), p.getDecision()));

            Function<ResourceInfo, ResourcesInfo.ResourceInfo> consResource = r -> new ResourcesInfo.ResourceInfo(
                    r.getNameI18n(lang).getOrElse(""), r.getEmail(), r.getResource().getType(), r.getOfficeId(), r.getOfficeStaffId(),
                    Option.empty(), Option.empty());

            return new EventBrief(event.getId(),
                    Option.when(canView, event.getEvent().getName()),
                    event.getParticipants().getOrganizerSafe().filter(o -> canView || forResource).flatMapO(consUser),
                    canView ? event.getParticipants().getAttendeesButNotOrganizerSafe().filterMap(consUser) : Cf.list(),
                    canView ? event.getParticipants().getOptionalAttendeesButNotOrganizerSafe().filterMap(consUser) : Cf.list(),
                    canView ? event.getResources().map(consResource) : Cf.list(),
                    event.getEvent().getPermAll() == EventActionClass.VIEW,
                    event.getEvent().getConferenceUrl());
        }));
    }

    private ListF<WebEventInfo> toWebEventInfos(Option<UserInfo> actorUserInfo, ListF<EventInstanceInfo> events,
                                                ListF<Layer> layers, Option<String> linkSignKey, DateTimeZone tz,
                                                Language lang, ActionInfo actionInfo) {
        Option<Office> actorUserOffice = actorUserInfo.filter(u -> u.getUid().isYandexTeamRu())
                .flatMapO(u -> officeManager.getDefaultOfficeForUser(u.getUid()));

        MapF<ParticipantId, WebUserInfo> webUserInfoById = Cf.hashMap();
        if (actorUserInfo.isPresent()) {
            MapF<ParticipantId, UserParticipantInfo> participantById = Cf.hashMap();

            events.forEach(instance -> {
                EventParticipants event = instance.getEventParticipants();

                Cf.<UserParticipantInfo>list().iterator()
                        .plus(event.getParticipants().getUserParticipantsSafe().iterator())
                        .plus(event.getParticipants().getNotDeclinedSubscribers().iterator().uncheckedCast())
                        .forEachRemaining(p -> participantById.put(p.getId(), p));
            });
            webUserInfoById.putAll(webNewUserManager.getWebUserInfos(
                    actorUserInfo.get().getUid(), Either.right(participantById.values()), lang));
        }

        MapF<Long, Layer> layerById = layers.toMapMappingToKey(Layer.getIdF());

        Function<Tuple3<EventInfoForPermsCheck, Option<EventUser>, Long>, EventActions> x = triple -> {
            Option<EventUser> eventUser = triple.get2();
            Option<Layer> layer = layerById.getO(triple.get3());

            return eventRoutines.getEventActions(actorUserInfo.get(), triple.get1(), eventUser, layer, actionInfo.getActionSource());
        };
        Function<Tuple3<EventInfoForPermsCheck, Option<EventUser>, Long>, EventActions> actionsF =
                actorUserInfo.isPresent() ? x.memoize() : t -> EventActions.empty();

        SetF<Long> confirmMainIds = Cf.hashSet();

        if (actorUserInfo.isPresent()) {
            ListF<Long> organizedIds = events.filterMap(e -> Option.when(
                    e.getParticipants().isOrganizer(actorUserInfo.get().getUid()), e.getMainEvent().getId()));

            confirmMainIds.addAll(repetitionConfirmationManager.findWhereTimeToConfirm(organizedIds, actionInfo));
        }

        return events.map(ei -> {
            {
                EventInfoForPermsCheck permInfo = ei.getInfoForPermsCheck();
                EventParticipants participants = ei.getEventParticipants();

                Function<UserParticipantInfo, WebUserInfo> userInfoF = p -> actorUserInfo.isPresent()
                        ? webUserInfoById.getOrThrow(p.getId())
                        : WebUserInfo.fromUserParticipantInfo(p);

                DateTime dt = ei.getInterval().getStart().toDateTime(tz);

                Option<WebUserParticipantInfo> organizer = Option.empty();
                ListF<WebUserParticipantInfo> attendees = Cf.arrayList();
                ListF<WebUserParticipantInfo> optionalAttendees = Cf.arrayList();
                ListF<WebResourceInfo> resources = webNewResourcesManager
                        .sortedByOffices(participants.getResources(), actorUserOffice, lang)
                        .map(r -> webNewResourcesManager.toWebResourceInfo(actorUserInfo, r, dt, lang));

                for (UserParticipantInfo participant: participants.getParticipants().getUserParticipantsSafe()) {
                    WebUserInfo info = userInfoF.apply(participant);

                    WebUserParticipantInfo webParticipant = new WebUserParticipantInfo(info, participant.getDecision());
                    if (participant.isOrganizer()) {
                        organizer = Option.of(webParticipant);
                    } else if (participant.isAttendee() && !participant.isOptional()) {
                        attendees.add(webParticipant);
                    } else if (participant.isAttendee() && participant.isOptional()){
                        optionalAttendees.add(webParticipant);
                    }
                }

                if (showCreatorAsOrganizer(ei.getParticipants(), layerById.getO(ei.getLayerId().get()), actorUserInfo.map(UserInfo::getUid))) {
                    organizer = getCreatorFromEvent(participants, ei.getEvent());
                }

                EventActions actions = actionsF.apply(Tuple3.tuple(permInfo, ei.getEventUser(), ei.getLayerId().get()));

                if (permInfo.isOrganizerLetToEditAnyMeeting()
                        && actorUserInfo.isMatch(info -> participants.getParticipants().isAttendee(info.getUid()))) {
                    actions = actions.withEdit(true);
                }

                val canAdminAllResources = actions.canEdit();

                val attachments = toWebAttachmentInfos(ei.getAttachmentsO());

                return WebEventInfo.create(
                        ei.getEventInterval(), ei.getMainEvent().getExternalId(),
                        ei.getEvent(), attachments, permInfo, ei.getRepetition(),
                        confirmMainIds.containsTs(ei.getMainEvent().getId()),
                        organizer, resources, attendees, optionalAttendees,
                        participants.getParticipants().getNotDeclinedSubscribers().map(userInfoF),
                        ei.getNotifications(NotificationsData.WEB_CHANNELS), actions,
                        ei.getEventUser().map(EventUser.getDecisionF()),
                        ei.getEventUser().map(EventUser.getAvailabilityF()),
                        ei.getLayerId(), tz, linkSignKey, canAdminAllResources);
            }
        });
    }

    public EventsCountInfo countEvents(
            Optional<PassportUid> actorUid,
            Optional<PassportUid> targetUid,
            ListF<Long> layerIds, Option<String> layerToken, Option<Boolean> layerToggledOn,
            LocalDate from, LocalDate till,
            Option<Boolean> opaqueOnly, Option<Boolean> showDeclined,
            Option<Boolean> hideAbsences, Optional<DateTimeZone> tz,
            ActionInfo actionInfo) {
        targetUid = targetUid.or(() -> actorUid);
        Supplier<Optional<DateTimeZone>> getUserTz = () -> actorUid.map(dateTimeManager::getTimeZoneForUid);
        val tzValue = tz.or(getUserTz).orElseThrow(() -> new IllegalArgumentException("Uid or tz required"));
        val actorUserInfo = userManager.getUserInfos(Option.x(actorUid)).singleO();

        val layers = webNewLayerManager.findLayersCanList(
                Option.x(targetUid), layerIds.map(Object::toString), layerToken, layerToggledOn,
                !hideAbsences.isSome(true), opaqueOnly.isSome(true)).get1();

        val tokenAccessedLayerId = layers
                .find(Layer.getPrivateTokenF().andThen(Cf2.isSomeOfF(layerToken)))
                .map(Layer::getId);

        val layerIdPredicate = LayerIdPredicate.list(layers.map(Layer::getId), false);

        val limits = EventLoadLimits.intersectsInterval(
                from.toDateTimeAtStartOfDay(tzValue).toInstant(),
                till.plusDays(1).toDateTimeAtStartOfDay(tzValue).toInstant());

        val filter = new EventsFilter(tokenAccessedLayerId, showDeclined.isSome(true), opaqueOnly.isSome(true), false);

        val events = eventRoutines.getSortedIndentsIMayView(actorUserInfo, layerIdPredicate, limits, filter, actionInfo.getActionSource());
        val viewStart = from.toDateTimeAtStartOfDay(tzValue).toInstant();
        val viewEnd = till.plusDays(1).toDateTimeAtStartOfDay(tzValue).toInstant();

        MapF<Long, MapF<LocalDate, AtomicInteger>> countsByLayerId = Cf.hashMap();

        for (EventIndentIntervalAndPerms indent : events) {
            val start = ObjectUtils.max(viewStart, indent.getEventInterval().getStart().toInstant(tzValue));
            val end = ObjectUtils.min(viewEnd, indent.getEventInterval().getEnd().toInstant(tzValue));

            for (InstantInterval interval : AuxDateTime.splitByDays(new InstantInterval(start, end), tzValue)) {
                countsByLayerId
                        .getOrElseUpdate(indent.getIndent().getLayerOrResourceId(), Cf::hashMap)
                        .getOrElseUpdate(new LocalDate(interval.getEnd(), tzValue), AtomicInteger::new)
                        .incrementAndGet();
            }
        }
        val layerUserByLayerId = targetUid
                .map(uid -> StreamEx.of(layerRoutines.getLayerUsersWithRelationsByUid(uid, Option.empty()))
                        .toMap(LayerUserWithRelations::getLayerId, identity()))
                .orElseGet(Collections::emptyMap);
        val missingIds = layers.map(Layer::getId).filterNot(layerUserByLayerId::containsKey);

        val layerUserByLayerIdWithMissing = missingIds.isEmpty()
                ? layerUserByLayerId
                : StreamEx.of(layerRoutines.getLayerUsersWithRelationsByLayerIds(missingIds))
                    .filter(LayerUserWithRelations::layerUserIsLayerCreator)
                    .mapToEntry(LayerUserWithRelations::getLayerId, identity())
                    .append(layerUserByLayerId)
                    .toImmutableMap();

        return new EventsCountInfo(layers.stream()
                .map(Layer::getId)
                .map(lid -> {
                    val color = layerUserByLayerIdWithMissing
                            .computeIfAbsent(lid, ignored -> { throw new IllegalStateException(); })
                            .getEvaluatedColor()
                            .printRgb();
                    val counts = countsByLayerId.getOrElse(lid, Cf.map()).mapValues(AtomicInteger::intValue);
                    return new EventsCountInfo.LayerCount(lid, color, counts);
                })
                .collect(CollectorsF.toList()));
    }

    public EventInfoShort handleReply(PassportUid uid, ReplyData replyData, ActionInfo actionInfo) {
        DateTimeZone tz = replyData.getTz().getOrElse(dateTimeManager.getTimeZoneForUidF().bind(uid));

        Option<Instant> instanceStart = replyData.getInstanceStartTs()
                .map(TimeUtils.localDateTime.toInstantF(DateTimeZone.UTC));

        boolean applyToAll = replyData.getApplyToAll().getOrElse(false);

        eventWebManager.handleEventInvitationDecision(
                uid, replyData.getEventId(), replyData.getPrivateToken(),
                instanceStart, applyToAll, replyData.toWebReplyData(), actionInfo);

        return new EventInfoShort(eventDao.findEventById(replyData.getEventId()), tz);
    }

    public RepetitionDescription getRepetitionDescription(
            PassportUid uid, RepetitionWithStartData repetitionData, Option<DateTimeZone> tzO, Language lang)
    {
        DateTimeZone tz = tzO.isPresent() ? tzO.get() : dateTimeManager.getTimeZoneForUid(uid);
        Instant start = repetitionData.getStartTs(tz);

        RepetitionInstanceInfo repetitionInfo = RepetitionInstanceInfo.create(
                new InstantInterval(start, start), tz,
                Option.of(repetitionData.getRepetition().toRepetition(start, tz)));

        return new RepetitionDescription(RepetitionToStringConverter.convert(repetitionInfo, lang));
    }

    public ParseTimeInEventNameResult parseTimeInEventName(String eventName, LocalDate date) {
        LocalDateTime startTs = date.toLocalDateTime(LocalTime.MIDNIGHT);
        boolean isAllDay = true;

        HumanReadableTimeParser.Result r = HumanReadableTimeParser.parse(eventName);

        if (r.gotSomething()) {
            startTs = r.getStartDateTime(date);
            isAllDay = r.isAllDay();

        }
        return new ParseTimeInEventNameResult(
                r.getRest(), startTs, isAllDay ? startTs.plusDays(1) : startTs.plusHours(1), isAllDay);
    }

    private ListF<WebUserInfo> toWebUserInfos(
            PassportUid uid, ListF<YandexUserParticipantInfo> participants, Language lang)
    {
        return webNewUserManager.getWebUserInfos(uid, Either.right(participants), lang).get2();
    }

    private ListF<WebUserParticipantInfo> toWebUserParticipantInfos(
            PassportUid uid, ListF<UserParticipantInfo> participants, Language lang)
    {
        final MapF<ParticipantId, WebUserInfo> infoById =
                webNewUserManager.getWebUserInfos(uid, Either.right(participants), lang).toMap();

        return participants.map(p -> new WebUserParticipantInfo(infoById.getOrThrow(p.getId()), p.getDecision()));
    }

    public MoveResourceEventsIds moveResource(PassportUid uid, long eventId, Optional<LocalDateTime> instanceStartTs,
                                              String fromResourceEmail, String toResourceEmail, ActionInfo actionInfo) {
        final var startEvent = instanceStartTs.map(localDateTime -> localDateTime.toDateTime(UTC).toInstant());
        val event = getInstance(Optional.of(uid), eventId, startEvent, actionInfo);

        val sourceEventData = constructEventData(startEvent, event, emptySet(), singleton(fromResourceEmail));
        val targetEventData = constructEventData(startEvent, event, singleton(toResourceEmail), emptySet());

        return eventWebManager.moveResource(userManager.getUserInfo(uid), sourceEventData, targetEventData, actionInfo);

    }

    public MoveResourceEventsIds moveResource(PassportUid uid, PassportUid sourceUid, long sourceEventId, long targetEventId,
                                              String resourceEmail, Optional<PassportUid> targetUidO,
                                              Optional<LocalDateTime> instanceStartTs, ActionInfo actionInfo) {
        final var startEvent = instanceStartTs.map(localDateTime -> localDateTime.toDateTime(UTC).toInstant());
        val sourceEvent = getInstance(Optional.of(sourceUid), sourceEventId, startEvent, actionInfo);
        val targetEvent = getInstance(targetUidO, targetEventId, startEvent, actionInfo);

        validateTimeSlotsEquality(startEvent, sourceEvent, targetEvent, actionInfo);
        val sourceUserInfo = userManager.getUserInfo(sourceUid);
        val eventAuthInfoForSourceUser = authorizer.loadEventInfoForPermsCheck(sourceUserInfo, sourceEvent);
        authorizer.ensureCanEditEvent(sourceUserInfo, eventAuthInfoForSourceUser, actionInfo.getActionSource());

        targetUidO.ifPresent(targetUid -> {
            val targetUserInfo = userManager.getUserInfo(targetUid);
            val eventAuthInfoForTargetUser = authorizer.loadEventInfoForPermsCheck(targetUserInfo, targetEvent);
            authorizer.ensureCanEditEvent(targetUserInfo, eventAuthInfoForTargetUser, actionInfo.getActionSource());
        });
        validateEventResource(sourceEvent, resourceEmail);

        val sourceEventData = constructEventData(startEvent, sourceEvent, emptySet(), singleton(resourceEmail));
        val targetEventData = constructEventData(startEvent, targetEvent, singleton(resourceEmail), emptySet());

        return eventWebManager.moveResource(userManager.getUserInfo(uid), sourceEventData, targetEventData, actionInfo);
    }

    private void validateTimeSlotsEquality(Optional<Instant> startEventO, EventWithRelations sourceEvent, EventWithRelations targetEvent,
                                           ActionInfo actionInfo) {
        startEventO.ifPresentOrElse(
            startEvent -> {
                checkEventInstanceStartTs(startEvent, sourceEvent.getId(), actionInfo);
                checkEventInstanceStartTs(startEvent, targetEvent.getId(), actionInfo);
            },
            () -> {
                ensureIsSingle(sourceEvent, "source");
                ensureIsSingle(targetEvent, "target");
                validateParameter(sourceEvent.getStartTs().isEqual(targetEvent.getStartTs()), "targetEvent", "Different time slot.");
            });
        validateParameter(sourceEvent.getEventDurationInMillis() == targetEvent.getEventDurationInMillis(), "targetEvent", "Different length.");
    }

    private EventWithRelations getInstance(Optional<PassportUid> uid, long eventId, Optional<Instant> startEvent,
                                           ActionInfo actionInfo) {
        return eventRoutines.getSingleInstance(Option.x(uid), Option.x(startEvent), true, false, eventId, actionInfo.getActionSource())
                .getEventWithRelations();
    }

    private void checkEventInstanceStartTs(Instant instantStart, long eventId, ActionInfo actionInfo) {
        val repetitionInstanceInfo = eventInfoDbLoader.getEventInfoById(Option.empty(), eventId, actionInfo.getActionSource()).getRepetitionInstanceInfo();
        validateParameter(RepetitionUtils.isValidStart(repetitionInstanceInfo, instantStart), "instanceStartTs",
                "Event (" + eventId + ") does not have instance for the given start timestamp.");
    }

    private static void ensureIsSingle(EventWithRelations eventWithRelations, String sideOfOperation) {
        validateParameter(!eventWithRelations.getEvent().getRepetitionId().isPresent(), sideOfOperation + "Event",
                "Is not single event and instance startTs is not set");
    }

    private static void validateEventResource(EventWithRelations event, String resourceEmail) {
        val resource = event.getResources().stream()
                .map(ResourceInfo::getEmail)
                .map(Email::getEmail)
                .filter(resourceEmail::equalsIgnoreCase)
                .findFirst();
        validateParameter(resource.isPresent(), "resource", "Source event does not have the resource");
    }

    private static List<EventInvitationData> prepareEventInvitationData(List<ParticipantInfo> participantInfos,
                                                                        Collection<String> resourcesToAdd, Set<String> resourcesToRemove) {
        return StreamEx.of(participantInfos)
                .map(ParticipantInfo::getEmail)
                .map(Email::getEmail)
                .append(resourcesToAdd)
                .remove(resourcesToRemove::contains)
                .distinct(String::toLowerCase)
                .map(Email::new)
                .map(EventInvitationData::new)
                .toImmutableList();
    }

    private static EventData constructEventData(Optional<Instant> startEvent, EventWithRelations event,
                                                Set<String> emailsToAdd, Set<String> emailsToRemove) {
        val eventData = new EventData();
        eventData.setInstanceStartTs(Option.x(startEvent));
        val participants = prepareEventInvitationData(event.getParticipants().getAllAttendeesSafe(), emailsToAdd, emailsToRemove);
        eventData.setInvData(new EventInvitationsData(Option.empty(), Cf.toArrayList(participants)));
        eventData.getEvent().setId(event.getId());
        return eventData;
    }

    private static void validateParameter(boolean value, String parameter, String message) {
        if (!value) {
            throw new IllegalParameterException(parameter, message);
        }
    }
}
