package ru.yandex.calendar.frontend.webNew;

import java.util.NoSuchElementException;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.calendar.frontend.bender.WebDateTime;
import ru.yandex.calendar.frontend.webNew.dto.out.Gap;
import ru.yandex.calendar.frontend.webNew.dto.out.NeedForRoomInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.UserGapInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.UserNeedForRoomInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.UserWorkMode;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.inside.passport.PassportUid;

public class WebNewGapsManager {
    @Autowired
    private UserManager userManager;
    @Autowired
    private WebNewLayerManager layoutManager;
    @Autowired
    private EventRoutines eventRoutines;

    public MapF<String, UserGapInfo> getUserGapsByUserLogins(LocalDateTime from, LocalDateTime to, ListF<String> userLogins, DateTimeZone tz) {
        Instant instantFrom = from.toDateTime(tz).toInstant();
        Instant instantTo = to.toDateTime(tz).toInstant();

        MapF<PassportUid, String> uidToLoginMap = associateUidsWithLoginsOrThrow(userLogins);
        ListF<PassportUid> userIds = uidToLoginMap.keys().toList();

        MapF<PassportUid, Option<Long>> userIdToAbsenceLayerIds = layoutManager.getUsersAbsenceLayerIds(userIds);
        MapF<PassportUid, ListF<Event>> userIdToEvent = userIdToAbsenceLayerIds.mapValuesWithKey(
                (uid, layerId) -> layerId.map(
                        l -> eventRoutines.findEventsByLayerIdInTimeRange(l, instantFrom, instantTo)
                                .filter(event -> event.getType() != EventType.USER)
                ).orElse(Cf.list())
        );

        return uidToLoginMap.mapValuesWithKey((uid, login) -> {
            ListF<Event> events = userIdToEvent.getOrElse(uid, Cf.list());
            SetF<Long> eventIds = events.map(Event::getId).toLinkedHashSet();
            ListF<EventUser> eventUsers = eventRoutines.findEventUsersByUidAndEventIds(uid, eventIds);

            String strWorkMode = userManager.getWorkMode(uid).orElse("office");
            UserWorkMode workMode = UserWorkMode.valueOfOrDefault(strWorkMode, UserWorkMode.OFFICE);

            ListF<Gap> gaps = events.map(event -> {
                EventUser eu = eventUsers.find(e -> e.getEventId() == event.getId()).get();

                return new Gap(
                        event.getId(),
                        event.getType(),
                        event.getName(),
                        WebDateTime.dateTime(event.getStartTs().toDateTime(tz)),
                        WebDateTime.dateTime(event.getEndTs().toDateTime(tz)),
                        event.getIsAllDay(),
                        eu.getDecision(),
                        eu.getAvailability()
                );
            });

            return new UserGapInfo(uid.getUid(), gaps, workMode);
        }).values().toMapMappingToKey(gap -> uidToLoginMap.getO(new PassportUid(gap.uid)).get());
    }

    public NeedForRoomInfo isRoomNeeded(LocalDateTime from, LocalDateTime to, ListF<String> userLogins, DateTimeZone tz, boolean isMixedAsRemote) {
        MapF<String, UserGapInfo> usersGaps = getUserGapsByUserLogins(from, to, userLogins, tz);

        MapF<String, UserNeedForRoomInfo> result = usersGaps.mapValuesWithKey((login, userGaps) ->
                new UserNeedForRoomInfo(isRoomNeededByGap(userGaps, isMixedAsRemote)));

        return new NeedForRoomInfo(result);
    }

    private MapF<PassportUid, String> associateUidsWithLoginsOrThrow(ListF<String> userLogins) {
        try {
            return userLogins.toMapMappingToKey(login -> {
                long uid = userManager.getYtUserUidByLogin(login)
                        .orElseThrow()
                        .getUid();
                return new PassportUid(uid);
            });
        } catch (NoSuchElementException e) {
            throw new IllegalParameterException("users", "each item of array must be existent login of staff-user");
        }
    }

    private boolean isRoomNeededByGap(UserGapInfo gapInfos, boolean isMixedAsRemote) {
        ListF<UserWorkMode> dontNeedRoomModes = Cf.arrayList(UserWorkMode.REMOTE);

        if (isMixedAsRemote) {
            dontNeedRoomModes.add(UserWorkMode.MIXED);
        }

        if (dontNeedRoomModes.containsTs(gapInfos.workMode)) {
            return gapInfos.gaps.find(g ->
                    g.isAllDay && g.type == EventType.OFFICE_WORK
            ).isNotEmpty();
        } else {
            return gapInfos.gaps.find(g ->
                    g.isAllDay && g.type != EventType.OFFICE_WORK && g.type != EventType.DUTY
            ).isEmpty();
        }
    }
}
