package ru.yandex.calendar.frontend.webNew;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.webNew.dto.in.ImportIcsData;
import ru.yandex.calendar.frontend.webNew.dto.in.LayerData;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerIdInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerOwnerInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerPrivateToken;
import ru.yandex.calendar.frontend.webNew.dto.out.LayersInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.StatusResult;
import ru.yandex.calendar.frontend.webNew.dto.out.WebUserInfo;
import ru.yandex.calendar.frontend.worker.task.IcsImportTask;
import ru.yandex.calendar.logic.beans.generated.IcsFeed;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerFields;
import ru.yandex.calendar.logic.beans.generated.LayerInvitation;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.beans.generated.LayerUserFields;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.avail.absence.AbsenceType;
import ru.yandex.calendar.logic.ics.feed.IcsFeedManager;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.layer.LayerInvitationManager;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.layer.LayerUserWithRelations;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.notification.NotificationDbManager;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.sending.bazinga.MessageExtraDao;
import ru.yandex.calendar.logic.sending.param.MessageExtra;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.perm.Authorizer;
import ru.yandex.calendar.logic.sharing.perm.LayerActionClass;
import ru.yandex.calendar.logic.sharing.perm.LayerInfoForPermsCheck;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.perm.LayerAction;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.color.Color;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.lang.Validate;

public class WebNewLayerManager {
    @Autowired
    private UserManager userManager;
    @Autowired
    private LayerRoutines layerRoutines;
    @Autowired
    private Authorizer authorizer;
    @Autowired
    private NotificationDbManager notificationDbManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private IcsFeedManager icsFeedManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private LayerInvitationManager layerInvitationManager;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private WebNewUserManager webNewUserManager;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private MessageExtraDao messageExtraDao;
    @Autowired
    private BazingaTaskManager bazingaTaskManager;
    @Value("${ics.import.max_events_size}")
    int maxEventsSize;

    public LayerIdInfo createLayer(final PassportUid uid, final LayerData data) {
        return transactionTemplate.execute(status -> {
            {
                long layerId = layerRoutines.createUserLayer(
                        uid, data.getNotifications(), data.getLayer(LayerType.USER, true),
                        data.getIsDefault().getOrElse(false), data.getLayerUser());

                layerInvitationManager.updateLayerSharing(
                        userManager.getUserInfo(uid), layerId,
                        data.getPermissionMapSafe(),
                        data.getIsEventsClosedByDefault().orElse(true),
                        getActionInfo());

                return new LayerIdInfo(layerId);
            }
        });
    }

    public StatusResult updateLayer(PassportUid uid, long layerId, boolean applyNotificationsToEvents, LayerData data) {
        return lockTransactionManager.lockAndDoInTransaction(LockResource.layerUser(uid), () -> {
            {
                val user = userManager.getUserInfo(uid);

                val layer = layerRoutines.getLayerById(layerId);
                layerRoutines.updateLayerUser(layerId, uid, data.getLayerUser(), getActionInfo());
                layerRoutines.updateNotification(
                        uid, layerId, NotificationsData.updateFromWeb(data.getNotifications()),
                        applyNotificationsToEvents, getActionInfo());

                val type = data.getTypeO().getOrElse(layerRoutines.getLayerById(layerId).getType());

                val source = getActionSource();
                val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                if (authorizer.canPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.EDIT, source)) {
                    Layer dataWithId = data.getLayer(type, false).copy();
                    dataWithId.setId(layerId);

                    layerRoutines.updateLayer(dataWithId, uid, getActionInfo());

                    if (data.getIsDefault().isPresent()) {
                        layerRoutines.updateDefaultLayer(uid, layerId, data.isDefault());
                    }

                    if ((type == LayerType.FEED) && data.getFeedUrlO().isPresent()) {
                        icsFeedManager.updateFeedUrl(uid, layerId, data.getFeedUrlO().get());
                    }
                }

                if ((type == LayerType.USER) && (data.getParticipants().isPresent())) {
                    val layerNewPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                    if (authorizer.canPerformLayerAction(user, layerNewPermInfo, Optional.empty(), LayerAction.GRANT, source)) {
                        layerInvitationManager.updateLayerSharing(
                                user, layerId, data.getPermissionMap(),
                                data.getIsEventsClosedByDefault().orElse(true), getActionInfo());
                    }
                }
                return StatusResult.ok();
            }
        });
    }

    public StatusResult toggleLayer(PassportUid uid, long layerId, boolean on) {
        layerRoutines.updateLayerUserVisibleInUi(uid, layerId, on);

        return StatusResult.ok();
    }

    public StatusResult deleteLayer(final PassportUid uid, final long layerId,
                                    final Option<Long> recipientLayerId, Option<Long> requestedNewDefaultLayerId) {
        return transactionTemplate.execute(status -> {
            {
                ListF<Long> ownUserLayers = getUserLayers(uid, Language.RUSSIAN).getOwnUserLayers();
                Option<Long> newDefaultLayerId = Option.empty();

                if (ownUserLayers.firstO().isSome(layerId)) {
                    newDefaultLayerId = requestedNewDefaultLayerId.isPresent()
                            && ownUserLayers.containsTs(requestedNewDefaultLayerId.get())
                            ? requestedNewDefaultLayerId
                            : ownUserLayers.drop(1).firstO();
                }

                val user = userManager.getUserInfo(uid);
                val layer = layerRoutines.getLayerById(layerId);

                val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                if (authorizer.canPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.DELETE, getActionSource())) {
                    if (recipientLayerId.isPresent()) {
                        layerRoutines.deleteMoveEvents(user, layerId, recipientLayerId.get(), getActionInfo());
                    } else {
                        layerRoutines.deleteLayer(user, layerId, getActionInfo());
                    }
                } else {
                    layerRoutines.detach(uid, uid, layerId, getActionInfo());
                }

                if (newDefaultLayerId.isPresent()) {
                    settingsRoutines.updateDefaultLayer(uid, newDefaultLayerId.get());
                }
                return StatusResult.ok();
            }
        });
    }

    public LayerInfo getLayer(PassportUid uid, long layerId, Language lang, Option<Function1B<? super LayerUserWithRelations>> filterFunction) {
        LayerUserWithRelations lu = layerRoutines.getLayerUserWithRelations(uid, layerId,
                        Option.of(lang))
                .filter(filterFunction.isPresent() ? filterFunction.get() : l -> canPerformLayerActionF(uid, LayerAction.LIST).apply(l.getLayer()))
                .getOrThrow(CommandRunException.createSituationF(
                        "layer not found by id " + layerId + " and uid " + uid,
                        Situation.NO_PERMISSIONS_FOR_LAYER_ACTION));

        DateTimeZone tz = dateTimeManager.getTimeZoneForUid(uid);

        long id = lu.getLayerId();
        String name = lu.getEvaluatedLayerName();
        String color = lu.getEvaluatedColor().printRgb();
        Option<Integer> defaultEventsDurationMinutes = lu.getLayerUser().getDefaultEventsDurationMinutes();

        boolean canAddEvent = lu.getLayer().getType() == LayerType.USER
                && canPerformLayerActionF(uid, LayerAction.CREATE_EVENT).apply(lu.getLayer());

        ListF<Notification> notifications = notificationDbManager.getNotificationsByLayerUserId(lu.getLayerUserId());
        boolean notifyAboutEventChanges = lu.getLayerUser().getIsNotifyChanges();
        boolean affectAvailability = lu.getLayerUser().getAffectsAvailability();

        boolean isOwner = lu.getLayer().getCreatorUid().sameAs(uid);

        val perm = lu.getLayerUser().getPerm();

        if (isOwner) {
            boolean isDefault = layerRoutines.getDefaultLayerId(uid).isSome(layerId);
            Option<String> token = lu.getLayer().getPrivateToken();

            Option<LayerInfo.FeedInfo> feedInfo = Option.empty();
            Option<IcsFeed> feed = icsFeedManager.findIcsFeedsByLayerIds(Cf.list(layerId)).single().get2();
            if (feed.isPresent()) {
                Option<LocalDateTime> lastUpdate = Option.when(
                        feed.get().getLastSuccessTs().isPresent(),
                        new LocalDateTime(feed.get().getLastSuccessTs().getOrElse(Instant.now()), tz));

                feedInfo = Option.of(new LayerInfo.FeedInfo(lastUpdate, feed.get().getUrl()));
            }

            val participants = prepareParticipants(uid, layerId, lang);
            boolean isEventsClosedByDefault = lu.getLayer().getIsEventsClosedByDefault();

            return new LayerInfo(
                    id, name, color, defaultEventsDurationMinutes.orElse(30), canAddEvent, isDefault, perm,
                    notifications, notifyAboutEventChanges, affectAvailability,
                    true, lu.getLayer().getType(), false, Option.empty(),
                    token, feedInfo, Option.of(isEventsClosedByDefault), Option.of(Cf.toList(participants)));

        } else {
            WebUserInfo owner = webNewUserManager.getWebUserInfoByParticipantId(
                    uid, ParticipantId.yandexUid(lu.getLayer().getCreatorUid()), lang);

            return new LayerInfo(
                    id, name, color, defaultEventsDurationMinutes.orElse(30), canAddEvent, false, perm,
                    notifications, notifyAboutEventChanges, affectAvailability,
                    false, lu.getLayer().getType(), true, Option.of(owner),
                    Option.empty(), Option.empty(),
                    Option.empty(), Option.empty());
        }
    }

    private List<LayerInfo.Participant> prepareParticipants(PassportUid uid, long layerId, Language lang) {
        val permsByLayerUsers = layerRoutines.getLayerPermissions(layerId);
        val invitations = layerInvitationManager.findLayerInvitations(layerId);
        val permissions = constructPermissionsMap(permsByLayerUsers, invitations);
        val userInfoById = webNewUserManager.getWebUserInfos(uid, permissions.keySet(), lang);
        return EntryStream.of(permissions)
                .mapKeys(userInfoById::get)
                .mapKeys(Objects::requireNonNull)
                .mapKeyValue(LayerInfo.Participant::new)
                .toImmutableList();
    }

    static Map<ParticipantId, LayerActionClass> constructPermissionsMap(Map<PassportUid, LayerActionClass> permsByLayerUsers,
                                                                        List<LayerInvitation> invitations) {
        val acceptedPermsByEmails = StreamEx.of(invitations)
                .mapToEntry(LayerInvitation::getEmail, LayerInvitation::getUid)
                .flatMapValues(Option::stream)
                .filterValues(permsByLayerUsers::containsKey)
                .mapValues(permsByLayerUsers::get)
                .toImmutableMap();
        return StreamEx.of(invitations)
                .mapToEntry(LayerInvitation::getEmail, LayerInvitation::getPerm)
                .mapToValue(acceptedPermsByEmails::getOrDefault)
                .mapKeys(ParticipantId::invitationIdForExternalUser)
                .toImmutableMap();
    }

    public LayersInfo getUserLayers(final PassportUid uid, Language lang) {
        ListF<LayerUserWithRelations> layerUsers = layerRoutines.getLayerUsersWithRelationsByUid(uid, Option.of(lang));

        layerUsers = layerUsers.filter(lu -> canPerformLayerActionF(uid, LayerAction.LIST).apply(lu.getLayer()));

        final Function1B<Layer> canCreateEventF = canPerformLayerActionF(uid, LayerAction.CREATE_EVENT);
        final MapF<Long, ListF<Notification>> notificationsByLayerUserId = notificationDbManager
                .getNotificationsByLayerUserIds(layerUsers.map(LayerUserWithRelations::getLayerUserId)).toMap();

        Option<Long> defaultLayerIdO = layerRoutines.getDefaultLayerId(uid);

        ListF<ParticipantId> participantIds =
                layerUsers.map(lu -> ParticipantId.yandexUid(lu.getLayer().getCreatorUid()));

        MapF<ParticipantId, WebUserInfo> owners =
                webNewUserManager.getWebUserInfos(uid, Either.left(participantIds), lang).toMap();

        MapF<Long, Integer> participantsCounts =
                layerInvitationManager.findLayerInvitationsCounts(layerUsers.map(LayerUserWithRelations::getLayerId));

        return new LayersInfo(layerUsers.map(lu -> {
            {
                long id = lu.getLayerId();
                String name = lu.getEvaluatedLayerName();

                boolean isToggledOn = lu.getLayerUser().getIsVisibleInUi();
                boolean canAddEvent = lu.getLayer().getType() == LayerType.USER && canCreateEventF.apply(lu.getLayer());
                val isEventsClosedByDefault = lu.getLayer().getIsEventsClosedByDefault();
                PassportUid ownerUid = lu.getLayer().getCreatorUid();
                boolean isOwner = ownerUid.sameAs(uid);

                Option<WebUserInfo> owner = isOwner ? Option.empty() : owners.getO(ParticipantId.yandexUid(ownerUid));

                boolean isDefault = defaultLayerIdO.isSome(id);

                Color color = lu.getEvaluatedColor();
                Option<Integer> defaultEventsDurationMinutes = lu.getLayerUser().getDefaultEventsDurationMinutes();
                ListF<Notification> notifications = notificationsByLayerUserId.getOrThrow(lu.getLayerUserId());

                Option<Integer> participantsCount = isOwner
                        ? participantsCounts.getO(id).orElse(Option.of(0))
                        : Option.empty();

                return new LayersInfo.LayerInfo(
                        id, name, isToggledOn, canAddEvent, isEventsClosedByDefault, isOwner, owner,
                        color, defaultEventsDurationMinutes.orElse(30), notifications, lu.affectsAvailability(),
                        isDefault, lu.getLayer().getType(), participantsCount);
            }
        }));
    }

    public Tuple2<ListF<Layer>, ListF<String>> findLayersCanList(
            Option<PassportUid> uid,
            ListF<String> layerStrIds, Option<String> layerToken, Option<Boolean> toggledOn,
            boolean includeAbsences, boolean opaqueOnly) {
        boolean requestedAbsences = layerStrIds.containsTs("absences");

        ListF<Long> layerIds = layerStrIds.filterMap(Cf.Long.parseSafeF());
        ListF<LayerType> layerTypes = requestedAbsences
                ? LayerType.R.valuesList().filter(LayerType::isAbsence)
                : Option.empty();

        ListF<Layer> found = layerStrIds.isNotEmpty() || layerToken.isPresent()
                ? layerRoutines.findLayersByIdsOrTypesOrTokens(layerIds, uid, layerTypes, layerToken)
                : layerRoutines.getLayerUsersWithRelationsByUids(uid).map(LayerUserWithRelations::getLayer);

        if ((toggledOn.isPresent() || opaqueOnly) && uid.isPresent()) {
            ListF<LayerUser> opaque = layerRoutines.getLayerUsers(SqlCondition.trueCondition()
                    .and(LayerUserFields.UID.eq(uid.get()))
                    .and(LayerUserFields.LAYER_ID.column().inSet(found.map(Layer::getId)))
                    .and(Option.when(opaqueOnly, LayerUserFields.AFFECTS_AVAILABILITY.eq(true)))
                    .and(Option.when(toggledOn.isPresent(), toggledOn.map(LayerUserFields.IS_VISIBLE_IN_UI::eq)::get)));

            found = found.filter(Layer.getIdF().andThen(opaque.map(LayerUser.getLayerIdF())::containsTs));
        }
        if (!includeAbsences && !requestedAbsences) {
            found = found.filterNot(l -> l.getType().isAbsence());
        }
        found = found.filter(canPerformLayerActionF(uid, LayerAction.LIST)
                .orF(l -> l.getPrivateToken().exists(layerToken::isSome)
                        && (uid.isPresent() || !l.getCreatorUid().isYandexTeamRu())));

        ListF<Long> foundIds = found.map(Layer::getId);
        ListF<String> foundTokens = found.flatMap(Layer::getPrivateToken);

        ListF<String> failedIds = layerStrIds.filterMap(s -> {
            Option<Long> id = Cf.Long.parseSafe(s);
            return Option.when(!s.equals("absences") && (!id.isPresent() || !foundIds.containsTs(id.get())), s);
        });
        failedIds = failedIds.plus(layerToken.filterNot(foundTokens.containsF()));
        failedIds = failedIds.plus(Option.when(requestedAbsences && !uid.isPresent(), "absences"));

        return Tuple2.tuple(found, failedIds);
    }

    public LayerPrivateToken obtainLayerPrivateToken(PassportUid uid, long layerId, boolean forceNew) {
        return new LayerPrivateToken(layerRoutines.obtainPtk(uid, layerId, forceNew));
    }

    public StatusResult handleLayerReply(PassportUid uid, String privateToken, Decision decision) {
        LayerInvitation invite = layerInvitationManager.getLayerInvitationByPrivateToken(privateToken, Option.of(uid));

        if (Decision.YES == decision) {
            if (layerRoutines.startNewSharing(uid, invite.getLayerId(), invite.getPerm())) {
                if (!invite.getUid().isSome(uid)) {
                    layerInvitationManager.createLayerInvitationIfAbsentForUser(
                            invite.getLayerId(), uid, invite.getPerm(), getActionInfo());
                }
            }
        } else if (Decision.NO == decision) {
            layerInvitationManager.removeLayerInvitation(uid, invite, getActionInfo());
        } else {
            throw new IllegalParameterException("decision", "Unexpected decision " + decision);
        }
        return StatusResult.ok();
    }

    public LayerInfo shareLayer(PassportUid uid, Option<Long> layerId, Option<String> privateToken, Language lang) {
        LayerActionClass action;
        final long id;

        if (privateToken.isPresent()) {
            Validate.isTrue(passportAuthDomainsHolder.containsYandexTeamRu(), "No token sharing in public");

            val layer = layerRoutines.getByPrivateToken(privateToken.get());
            id = layer.getId();
            authorizer.ensureLayerType(LayerInfoForPermsCheck.fromLayer(layer), LayerType.USER);
            action = LayerActionClass.VIEW;
        } else {
            val user = userManager.getUserInfo(uid);
            id = layerId.get();
            val layer = layerRoutines.getLayerById(id);
            val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
            authorizer.ensureLayerType(layerPermInfo, LayerType.USER);
            authorizer.ensureCanPerformLayerAction(user, layerPermInfo, Optional.empty(), LayerAction.LIST, ActionSource.WEB);
            action = LayerActionClass.ACCESS;
        }

        if (!layerRoutines.startNewSharing(uid, id, action)) {
            throw CommandRunException.createSituation(
                    "It's pointless for user (" + uid + ") to share layer (id = " + id + ")",
                    Situation.ALREADY_SHARED
            );
        }
        if (action != LayerActionClass.ACCESS) {
            layerInvitationManager.createLayerInvitationIfAbsentForUser(id, uid, action, getActionInfo());
        }

        return getLayer(uid, id, lang, Option.empty());
    }

    public LayerInfo createIcsFeed(PassportUid uid, String url, Language lang) {
        IcsFeed feed = icsFeedManager.subscribe(uid, url, "Calendar");

        return getLayer(uid, feed.getLayerId(), lang, Option.empty());
    }

    public StatusResult forceFeedUpdate(final PassportUid uid, final long feedId) {
        icsFeedManager.forceIcsFeedUpdate(uid, feedId);
        return StatusResult.ok();
    }

    public LayerInfo importIcs(PassportUid uid, ImportIcsData data) {
        Optional<IcsCalendar> parsed = ImportIcsData.getIcs(data.getIcsWrapper().toOptional());
        ValidateParam.isTrue("data", parsed.isPresent() != data.getUrl().isPresent(), "Specify either url or ics");
        ValidateParam.isFalse("url", data.isFeed() && !data.getUrl().isPresent(), "Url should be specified for feed");
        ValidateParam.isFalse("layer", !data.isFeed() && !data.isNew() && !data.getLayerId().isPresent(), "Invalid layer reference");
        IcsCalendar ics = parsed.orElseGet(() -> icsFeedManager.downloadAndParseNow(data.getUrl().get()));

        if (ics.getEvents().size() > maxEventsSize && !data.isFeed()) {
            throw new IllegalParameterException("data", "Too many events");
        }

        Layer layerData = new Layer();
        ics.getXWrCalname().filterNot(String::isEmpty).forEach(layerData::setName);
        data.getLayerData().getName().filterNot(String::isEmpty).forEach(layerData::setName);

        if (data.isFeed()) {
            layerData.setFieldValueDefault(LayerFields.NAME, "Calendar");

            IcsFeed feed = transactionTemplate.execute(s -> {
                long layerId = layerRoutines.createLayerWithLayerUser(
                        uid, LayerType.FEED, layerData,
                        data.getLayerData().getLayerUser(), data.getLayerData().getNotifications());

                return icsFeedManager.subscribe(uid, data.getUrl().get(), layerId);
            });
            return getLayer(uid, feed.getLayerId(), Language.RUSSIAN, Option.empty());

        } else {
            data.getLayerData().getIsEventsClosedByDefault().ifPresent(layerData::setIsEventsClosedByDefault);

            long layerId = data.isNew()
                    ? layerRoutines.createUserLayer(
                    uid, data.getLayerData().getNotifications(), layerData,
                    data.getLayerData().getIsDefault().getOrElse(false), data.getLayerData().getLayerUser())
                    : data.getLayerId().get();

            long extraId = messageExtraDao.save(new MessageExtra(ics));

            bazingaTaskManager.schedule(new IcsImportTask(uid, extraId, layerId));

            return getLayer(uid, layerId, Language.RUSSIAN, Option.empty());
        }
    }

    private Function1B<Layer> canPerformLayerActionF(PassportUid uid, LayerAction action) {
        return canPerformLayerActionF(Option.of(uid), action);
    }

    private Function1B<Layer> canPerformLayerActionF(Option<PassportUid> uid, final LayerAction action) {
        Function0<Option<UserInfo>> userInfo = Cf2.f0(() -> userManager.getUserInfos(uid).singleO()).memoize();
        val actionSource = getActionSource();

        return layer -> userInfo.apply()
                .map(user -> {
                    val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                    return authorizer.canPerformLayerAction(user, layerPermInfo, Optional.empty(), action, actionSource);
                })
                .getOrElse(() -> {
                    val layerPermInfo = LayerInfoForPermsCheck.fromLayer(layer);
                    return authorizer.canPerformLayerAction(layerPermInfo, action, actionSource);
                });
    }

    private ActionInfo getActionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }

    private ActionSource getActionSource() {
        return getActionInfo().getActionSource();
    }

    public LayerOwnerInfo getOwnerByLayerId(long layerId) {
        val layer = layerRoutines.findLayerById(layerId).toOptional()
                .orElseThrow(() -> new IllegalParameterException("layerId", "There is no layer with such layerId"));
        val uid = layer.getCreatorUid();
        val login = settingsRoutines.getSettingsByUid(uid).getUserLogin().toOptional()
                .orElseThrow(() -> new IllegalParameterException("layerId", "Unable to find login for owner of the layer"));
        return new LayerOwnerInfo(uid.getUid(), login);
    }

    public MapF<PassportUid, Option<Long>> getUsersAbsenceLayerIds(ListF<PassportUid> userIds) {
        return userIds.toMapMappingToValue(userId -> layerRoutines.getAbsenceLayerIdIfPresent(userId, AbsenceType.ABSENCE));
    }
}
