package ru.yandex.calendar.frontend.webNew;

import java.util.LinkedHashMap;
import java.util.Optional;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.bender.WebDate;
import ru.yandex.calendar.frontend.bender.WebDateTime;
import ru.yandex.calendar.frontend.bender.WebDateUtils;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailParameters;
import ru.yandex.calendar.frontend.webNew.dto.in.IntervalAndRepetitionData;
import ru.yandex.calendar.frontend.webNew.dto.out.MoveResourceEventsIds;
import ru.yandex.calendar.frontend.webNew.dto.out.OfficesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.OfficesTzInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ResourcesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ResourcesSchedule;
import ru.yandex.calendar.frontend.webNew.dto.out.SubjectAvailability;
import ru.yandex.calendar.frontend.webNew.dto.out.UserOrResourceInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.UsersAndResourcesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.WebResourceInfo;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.resource.MultiofficeResourceFilter;
import ru.yandex.calendar.logic.resource.OfficeFilter;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.resource.SpecialResources;
import ru.yandex.calendar.logic.resource.schedule.EventIdOrReservationInterval;
import ru.yandex.calendar.logic.resource.schedule.ResourceDaySchedule;
import ru.yandex.calendar.logic.resource.schedule.ResourceEventsAndReservations;
import ru.yandex.calendar.logic.resource.schedule.ResourceScheduleManager;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateInterval;
import ru.yandex.calendar.util.dates.DateOrDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox.PassportDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;

@Slf4j
public class WebNewResourcesManager {
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private ResourceComparators resourceComparators;
    @Autowired
    private ResourceScheduleManager resourceScheduleManager;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private WebNewAvailabilityManager webNewAvailabilityManager;
    @Autowired
    private EventWebManager eventWebManager;
    @Autowired
    private WebNewEventManager webNewEventManager;
    @Autowired
    private WebNewUserManager webNewUserManager;
    @Autowired
    private UserManager userManager;

    public ResourcesInfo findResourcesByStaffIds(
            PassportUid uid,
            ListF<Long> officeStaffIds,
            Option<String> filter,
            Option<String> resourceNamePrefix,
            final Language lang
    ) {
        ListF<Long> officeIds = internalOfficeIdsByStaffIdsInSameOrder(officeStaffIds);
        MultiofficeResourceFilter resourceFilter = new MultiofficeResourceFilter(officeIds, filter, resourceNamePrefix, Cf.list());

        return findResources(uid, resourceFilter, lang);
    }

    public ResourcesInfo findResources(PassportUid uid, MultiofficeResourceFilter filter, final Language lang) {
        val resources = findResourcesInner(uid, filter, lang);
        val userInfo = userManager.getUserInfo(uid);

        return new ResourcesInfo(resources.map(r -> new ResourcesInfo.ResourceInfo(
                r.getNameI18n(lang).getOrElse(""), r.getEmail(),
                r.getResource().getType(), r.getOfficeId(), r.getOfficeStaffId(), Option.empty(),
                Option.when(userInfo.canAdminResource(r.getResource()), true))), resources.size());
    }

    public ResourcesInfo findAvailableResourcesByStaffIds(
            PassportUid uid,
            ListF<Long> officeStaffIds,
            ListF<String> filters,
            Option<String> query,
            ListF<Email> emails,
            IntervalAndRepetitionData interval,
            int limit,
            AvailParameters params,
            Language lang
    ) {
        ListF<Long> officeIds = internalOfficeIdsByStaffIdsInSameOrder(officeStaffIds);
        MultiofficeResourceFilter resourceFilter = new MultiofficeResourceFilter(officeIds, filters, query, emails);

        return findAvailableResources(uid, resourceFilter, interval, limit, params, lang);
    }

    public ResourcesInfo findAvailableResources(
            PassportUid uid, MultiofficeResourceFilter filter,
            IntervalAndRepetitionData interval, int limit, AvailParameters params, Language lang) {
        val tz = params.tz.getOrElse(() -> dateTimeManager.getTimeZoneForUid(uid));
        val repeatTz = eventWebManager.getEventTimezoneIfExists(params.exceptEventId).getOrElse(tz);
        val repetitionInfo = interval.toRepetitionInstanceInfo(tz, repeatTz);

        ListF<ResourceInfo> resources = findResourcesInner(uid, filter, lang);

        val singleEmails = resources.groupBy(ResourceInfo::getOfficeId).entrySet()
                .filterMap(e -> Option.when(e.getValue().size() == 1, e.getValue().first().getEmail()));

        if (!userManager.isYamoneyUser(uid)) {
            resources = resources.filter(r -> r.getResource().getType() != ResourceType.YAMONEY_ROOM
                    || singleEmails.containsTs(r.getEmail()));
        }

        val resourceByEmail = resources.toMapMappingToKey(ResourceInfo::getEmail);

        val officesResources = resources.groupBy(ResourceInfo::getOfficeId).values();

        MapF<Long, ListF<SubjectAvailability>> availsByOfficeId = Cf.x(new LinkedHashMap<>());

        resources.forEach(r -> availsByOfficeId.computeIfAbsent(r.getOfficeId(), o -> Cf.arrayListWithCapacity(limit)));

        for (ListF<ResourceInfo> ress : Cf.list(
                officesResources.flatMap(l -> l.take(limit * 3)), officesResources.flatMap(l -> l.drop(limit * 3)))) {
            final var emails = ress.filterMap(res ->
                    Option.when(availsByOfficeId.getTs(res.getOfficeId()).size() < limit, res.getEmail()));

            val avails = webNewAvailabilityManager
                    .getAvailabilities(uid, emails, Either.left(repetitionInfo), params, true, tz)
                    .filter(avail -> avail.isAvailable() || singleEmails.containsTs(avail.getEmail()));

            avails.forEach(avail -> {
                val list = availsByOfficeId.getTs(
                        resourceByEmail.getTs(avail.getEmail()).getOfficeId());

                if (list.size() < limit) {
                    list.add(avail);
                }
            });
        }

        val userInfo = userManager.getUserInfo(uid);

        return new ResourcesInfo(availsByOfficeId.values().iterator().flatMap(ListF::iterator).map(avail -> {
            val resource = resourceByEmail.getTs(avail.getEmail());

            return new ResourcesInfo.ResourceInfo(
                    resource.getNameI18n(lang).getOrElse(""),
                    resource.getEmail(),
                    resource.getResource().getType(),
                    resource.getOfficeId(),
                    resource.getOfficeStaffId(),
                    Option.of(singleEmails.containsTs(resource.getEmail())
                            ? avail.getAvailabilityInfo()
                            : avail.getAvailabilityInfo().withoutValue()),
                    Option.when(userInfo.canAdminResource(resource.getResource()), true));
        }).toList(), resources.size());
    }

    private ListF<ResourceInfo> findResourcesInner(PassportUid uid, MultiofficeResourceFilter filter, Language lang) {
        val noPrefixRequest = filter.getResourceNamePrefix().getOrElse("").isEmpty();

        val officeIds = filter.getOfficeIds();

        Comparator<ResourceInfo> officeComparator = officeIds.size() > 1
                ?
                Cf2.f(officeIds.zipWithIndex().toMap()::getTs).compose(ResourceInfo::getOfficeId).andThenNaturalComparator()
                : Comparator.constEqualComparator();

        Comparator<ResourceInfo> nameComparator = ResourceInfo.getNameI18nF(lang)
                .andThen(Cf2.f(us -> us.getOrElse(""))).andThenNaturalComparator();

        Function1B<ResourceInfo> isNoSuggestRoomF = r -> r.getResource().getExchangeName()
                .exists(SpecialResources.noSuggestInRooms.get().containsF());

        final Function1B<ResourceInfo> matcher;
        final Comparator<ResourceInfo> comparator;

        if (!noPrefixRequest) {
            matcher = isNoSuggestRoomF.notF().orF(ResourceInfo.resourceF().andThen(filter.nameWordMatchesPrefixF()));
            comparator = officeComparator.thenComparing(nameComparator);

        } else {
            matcher = isNoSuggestRoomF.notF();
            comparator = officeComparator.thenComparing(
                    resourceComparators.distanceFromUserComparator(uid)).thenComparing(nameComparator);
        }
        val officeFilter = officeIds.isNotEmpty() ? OfficeFilter.any().withId(officeIds) : OfficeFilter.any();

        val resources = filter.isOfficeSpecific()
                ? resourceRoutines.getDomainResourcesCanBookWithLayersAndOffices(uid, filter.getFiltersByOffice())
                : resourceRoutines.getDomainResourcesCanBookWithLayersAndOffices(uid, officeFilter, filter.getFilter());

        return resources.filter(matcher).sorted(comparator);
    }

    public ListF<MoveResourceEventsIds> moveResourcesSchedules(
            long fromResourceId, long toResourceId,
            String fromDateStr, int daysCountOptional) {
        WebDate fromDate = WebDateUtils.parseDate(fromDateStr);
        final ListF<Long> ids = Cf.arrayList(fromResourceId, toResourceId);
        MapF<Long, Resource> resourceById =
                resourceRoutines.getResourcesByIds(ids).toMapMappingToKey(Resource.getIdF());
        if (resourceById.size() != 2) {
            throw new IllegalArgumentException(String.format("Can't find resources with ids: fromResourceId = %s, " +
                    "toResourceId = %s", fromResourceId, toResourceId));
        }
        final String fromResourceEmail =
                ResourceRoutines.getResourceEmail(resourceById.getOrThrow(fromResourceId)).getEmail();
        final String toResourceEmail =
                ResourceRoutines.getResourceEmail(resourceById.getOrThrow(toResourceId)).getEmail();

        Instant currentInstant = fromDate.toInstant(DateTimeZone.UTC);
        final ListF<MoveResourceEventsIds> result = Cf.arrayList();
        for (int i = 0; i <= daysCountOptional; i++) {
            final ListF<LocalDate> days = Cf.arrayList(currentInstant.toDateTime(DateTimeZone.UTC).toLocalDate());
            final ListF<ResourceDaySchedule> resourceScheduleDataForDays =
                    resourceScheduleManager.getResourceScheduleDataForDays(Option.empty(), resourceById.keys(), days,
                            DateTimeZone.UTC, Cf.list(), ActionInfo.adminManager());
            ResourceDaySchedule scheduleForFrom = resourceScheduleDataForDays
                    .filter(resourceDaySchedule -> resourceDaySchedule.getResourceId() == fromResourceId).first();
            ResourceDaySchedule scheduleForTo = resourceScheduleDataForDays
                    .filter(resourceDaySchedule -> resourceDaySchedule.getResourceId() == toResourceId).first();

            final ResourceEventsAndReservations resourceEventsAndReservations =
                    scheduleForFrom.getSchedule().filterIntervals(eventIdOrReservationInterval -> {
                        for (EventIdOrReservationInterval interval : scheduleForTo.getSchedule().getIntervals()) {
                            if (eventIdOrReservationInterval.getInterval().contains(interval.getInterval())) {
                                log.debug("Can't move resource for event: " + eventIdOrReservationInterval.getEventId().get()
                                        + " from " + fromResourceEmail + " -> " + toResourceEmail);
                                return false;
                            }
                        }
                        return true;
                    });
            final ListF<Long> eventIds = resourceEventsAndReservations.getEventIds();
            for (Long eventId : eventIds) {
                log.debug("moving resource for event: " + eventId + " from " + fromResourceEmail + " -> " + toResourceEmail);
                result.addAll(webNewEventManager.moveResource(
                        ResourceRoutines.getMasterOfResources(PassportDomain.YANDEX_TEAM_RU).toUidOrZero().toUid(),
                        eventId, Optional.empty(), fromResourceEmail, toResourceEmail,
                        ActionInfo.adminManager().withActionSource(ActionSource.DB_REPAIRER)));
            }
            currentInstant = currentInstant.plus(Duration.standardDays(1));
        }
        return result;
    }

    public ResourcesSchedule getResourcesSchedule(
            PassportUid uid, MultiofficeResourceFilter resourceFilters,
            WebDate from, WebDate till, boolean bookableOnly,
            Option<Long> eventId, Option<DateTimeZone> tzO, Language lang) {
        Function2<PassportUid, Tuple2List<Long, ResourceFilter>, ListF<ResourceInfo>> getF = bookableOnly
                ? resourceRoutines::getDomainResourcesCanBookWithLayersAndOffices
                : resourceRoutines::getDomainResourcesCanViewWithLayersAndOffices;

        val officeIds = resourceFilters.getOfficeIds();
        val offices = officeManager.getOfficesByIds(officeIds);

        val tz = tzO
                .orElse(() -> Option.when(offices.size() == 1, () -> OfficeManager.getOfficeTimeZone(offices.single())))
                .getOrElse(() -> dateTimeManager.getTimeZoneForUid(uid));

        val resources = getF.apply(uid, resourceFilters.getFiltersByOffice());

        InstantInterval interval = new InstantInterval(from.toInstant(tz), till.toInstantNextDay(tz));

        val comparator = ResourceInfo.resourceF().andThen(
                ResourceFields.FLOOR_NUM.getF().andThenNaturalComparator().thenComparing(
                        ResourceComparators.BY_GROUP_NAME.uncheckedCastC()).thenComparing(
                        ResourceFields.POS.getF().andThenNaturalComparator()).thenComparing(
                        ResourceFields.NAME.getF().andThenNaturalComparator()).uncheckedCastC());

        val userInfo = userManager.getUserInfo(uid);

        Function<Tuple2<ResourceInfo, ResourceEventsAndReservations>, ResourcesSchedule.Resource> toResource = rs ->
                toResourceScheduleResource(userInfo, rs.get1(), rs.get2(), interval, bookableOnly, tz, lang);

        val schedule = resourceScheduleManager
                .getResourceScheduleDataForInterval(Option.of(uid), resources, interval, tz, eventId, getActionInfo())
                .sortedBy1(Cf2.f(officeIds.zipWithIndex().toMap()::getTs).compose(ResourceInfo::getOfficeId)
                        .andThenNaturalComparator().thenComparing(comparator));

        return new ResourcesSchedule(
                Cf2.stableGroupBy(schedule, t -> t.get1().getOfficeId()).mapEntries((id, rs) -> {
                    val office = rs.first().get1().getOffice();

                    return new ResourcesSchedule.Office(office.getId(), office.getStaffId(),
                            ResourceRoutines.getNameI18n(office, lang),
                            OfficeManager.getOfficeTimeZone(office).getID(), rs.map(toResource));
                }));
    }

    public ResourcesSchedule getResourcesScheduleByStaffIds(
            PassportUid uid,
            ListF<Long> officeStaffIds,
            ListF<String> filters,
            Option<String> query,
            ListF<Email> emails,
            WebDate from,
            WebDate till,
            boolean bookableOnly,
            Option<Long> eventId,
            Option<DateTimeZone> tzO,
            Language lang
    ) {
        ListF<Long> officeIds = internalOfficeIdsByStaffIdsInSameOrder(officeStaffIds);
        MultiofficeResourceFilter resourceFilter = new MultiofficeResourceFilter(officeIds, filters, query, emails);

        return getResourcesSchedule(uid, resourceFilter, from, till, bookableOnly, eventId, tzO, lang);
    }

    public WebResourceInfo getResourceDescription(PassportUid uid, Email email, Language lang) {
        return toWebResourceInfo(
                Option.of(userManager.getUserInfo(uid)),
                resourceRoutines.findByExchangeEmail(email).get(), Option.empty(), lang);
    }

    public UserOrResourceInfo getUserOrResourceInfo(PassportUid uid, Email email, Language lang) {
        val id = eventInvitationManager.getParticipantIdByEmail(email);

        if (id.isResource()) {
            return UserOrResourceInfo.resource(getWebResourceInfosByIds(
                    userManager.getUserInfo(uid), Cf.list(id.getResourceId()), lang).single().get2());
        } else {
            return UserOrResourceInfo.user(webNewUserManager.getWebUserInfoByParticipantId(uid, id, lang));
        }
    }

    public OfficesInfo getOffices(
            PassportUid uid, Option<Boolean> includeOthers, Option<DateTimeZone> tzO, Language lang) {
        val domain = settingsRoutines.getPassportDomain(uid);
        val tz = tzO.getOrElse(() -> dateTimeManager.getTimeZoneForUid(uid));

        ListF<Office> offices = officeManager.getDomainOfficesWithAtLeastOneActiveResource(uid, domain);
        val userOffice = officeManager.getDefaultOfficeForUser(uid);

        if (!includeOthers.isSome(true)) {
            offices = offices.filterNot(WebNewResourcesManager::isOther);
        }
        val officesSorted = sortOfficesInPreferredOrder(offices, userOffice, lang);
        return new OfficesInfo(officesSorted.map(o -> toOfficeInfo(o, tz, lang)));
    }

    public OfficesTzInfo getOfficesTzOffsets(PassportUid uid, LocalDateTime ts, Option<DateTimeZone> tzO) {
        val tz = tzO.getOrElse(() -> dateTimeManager.getTimeZoneForUid(uid));
        val instant = AuxDateTime.toInstantIgnoreGap(ts, tz);

        val domain = settingsRoutines.getPassportDomain(uid);
        val offices = officeManager.getDomainOfficesWithAtLeastOneActiveResource(uid, domain);

        return new OfficesTzInfo(offices.map(o -> {
            val officeTz = OfficeManager.getOfficeTimeZone(o);
            return new OfficesTzInfo.OfficeTzInfo(o.getId(), officeTz.getOffset(instant) - tz.getOffset(instant));
        }));
    }

    private Option<Email> parseLoginOrEmail(String loginOrEmail, Option<PassportDomain> domain) {
        loginOrEmail = loginOrEmail.toLowerCase().trim();
        if (loginOrEmail.isEmpty()) {
            return Option.empty();
        }
        Option<Email> parsed = Option.empty();
        try {
            parsed = Email.parseSafe(new InternetAddress(loginOrEmail).getAddress());
        } catch (AddressException ignored) {
        }

        if (parsed.isPresent()) {
            return parsed;
        }
        parsed = Emails.punycodeSafe(loginOrEmail);
        if (parsed.isPresent()) {
            return parsed;
        }
        if (domain.isPresent() && domain.get().sameAs(PassportDomain.YANDEX_TEAM_RU)) {
            parsed = Option.x(userManager.getYtUserEmailByLogin(loginOrEmail.endsWith("@")
                    ? loginOrEmail.substring(0, loginOrEmail.length() - 1)
                    : loginOrEmail));
        }
        if (parsed.isPresent()) {
            return parsed;
        }
        if (domain.isPresent() && loginOrEmail.endsWith("@")) {
            parsed = Email.parseSafe(loginOrEmail + domain.get().getDomain());
        }
        return parsed;
    }

    private Function<String, Option<Email>> parseLoginOrEmailF(final Option<PassportDomain> domain) {
        return s -> parseLoginOrEmail(s, domain);
    }

    public UsersAndResourcesInfo findUsersAndResources(PassportUid uid, String loginOrEmailsStr, Language lang) {
        val domain = resourceRoutines.getDomainByUidUnlessPublic(uid);
        val loginOrEmails = Cf.x(loginOrEmailsStr.split(",|;"));

        val parsed = loginOrEmails.zipWith(parseLoginOrEmailF(domain));

        val notParsed = parsed.filterBy2Not(Option::isPresent).get1();
        val emails = Cf2.flatBy2(parsed).get2().stableUnique();

        val ids = eventInvitationManager.getParticipantIdsByEmails(emails).get2();
        val resourceIds = ids.filterMap(ParticipantId.getResourceIdIfResourceF());
        val userIds = ids.filter(ParticipantId.isResourceF().notF());

        val resources = getWebResourceInfosByIds(userManager.getUserInfo(uid), resourceIds, lang).get2();
        val users = webNewUserManager.getWebUserInfos(uid, Either.left(userIds), lang).get2();

        return new UsersAndResourcesInfo(users, resources, notParsed);
    }

    public Tuple2List<Long, WebResourceInfo> getWebResourceInfosByIds(UserInfo userInfo, ListF<Long> ids,
                                                                      Language lang) {
        MapF<Long, Resource> resourceById =
                resourceRoutines.getResourcesByIds(ids).toMapMappingToKey(Resource.getIdF());

        return ids.zipWith(id -> toWebResourceInfo(Option.of(userInfo), resourceById.getOrThrow(id), Option.empty(),
                lang));
    }

    public WebResourceInfo toWebResourceInfo(
            Option<UserInfo> userInfo, ResourceInfo resource, DateTime dt, Language lang) {
        val officeTz = OfficeManager.getOfficeTimeZone(resource.getOffice());
        val tzOffset = officeTz.getOffset(dt.getMillis()) - dt.getZone().getOffset(dt.getMillis());

        return toWebResourceInfo(userInfo, resource.getResource(), Option.of(tzOffset), lang);
    }

    public WebResourceInfo toWebResourceInfo(
            Option<UserInfo> userInfo, Resource resource, Option<Integer> tzOffset, Language lang) {
        val canAdmin = userInfo.isPresent() && userInfo.get().canAdminResource(resource);
        val office = officeManager.getOfficeById(resource.getOfficeId());

        return new WebResourceInfo(
                resource.getOfficeId(),
                ResourceRoutines.getNameI18n(resource, lang).getOrElse(""),
                ResourceRoutines.getAlterNameI18n(resource, lang).getOrElse(""),
                ResourceRoutines.getResourceEmail(resource),
                resource.getPhone().map(Cf.Integer.toStringF()).getOrElse(""),
                resource.getVideo().map(Cf.Integer.toStringF()).getOrElse(""), resource.getDescription(),
                resource.getSeats().getOrElse(0), resource.getCapacity().getOrElse(0), resource.getVoiceConferencing(),
                resource.getProjector(), resource.getLcdPanel(), resource.getMarkerBoard(),
                resource.getDesk(), resource.getGuestWifi(),
                resource.getFloorNum(), resource.getMapUrl(), resource.getIsActive(), resource.getType(),
                Option.when(!canAdmin && SpecialResources.isRepetitionUnacceptable(resource), true),
                Option.when(canAdmin, true),
                ResourceRoutines.getCityNameI18n(office, lang),
                Option.of(ResourceRoutines.getNameI18n(office, lang)),
                ResourceRoutines.getGroupNameI18n(resource, lang), tzOffset);
    }

    public ResourcesSchedule.Resource toResourceScheduleResource(
            UserInfo userInfo, ResourceInfo resourceInfo,
            ResourceEventsAndReservations resourceEvents, InstantInterval scheduleInterval,
            boolean isBookableOnly, DateTimeZone tz, Language lang
    ) {
        val reservationCreators = resourceEvents.getIntervals()
                .filterMap(EventIdOrReservationInterval::getReservationCreatorUid).map(ParticipantId::yandexUid);

        val authorInfoById = webNewUserManager
                .getWebUserInfos(userInfo.getUid(), Either.left(reservationCreators), lang)
                .toMap();

        final var events = resourceEvents.getIntervals()
                .map(i -> new ResourcesSchedule.ResourceEvent(i.getEventId(), i.getReservationId(),
                        i.getReservationCreatorUid().map(ParticipantId::yandexUid).filterMap(authorInfoById::getO),
                        WebDateTime.dateTime(i.getInterval().getStart().toDateTime(tz)),
                        WebDateTime.dateTime(i.getInterval().getEnd().toDateTime(tz))));

        val resource = resourceInfo.getResource();
        val resourceTz = OfficeManager.getOfficeTimeZone(resourceInfo.getOffice());

        final var reachedDistance = SpecialResources.getEventMaxStart(userInfo, resourceInfo, Instant.now())
                .filterNot(scheduleInterval.getEnd()::isBefore)
                .map(d -> new DateInterval(Option.of(DateOrDateTime.dateTime(d.toLocalDateTime())), Option.empty()));

        val restrictions = reachedDistance.plus(SpecialResources.getRestrictionDates(resourceInfo)
                .filter(d -> d.toInstantInterval(resourceTz).overlaps(scheduleInterval)));

        val canBook = userInfo.canBookResource(resource);

        val canAdmin = Option.of(userInfo.canAdminResource(resource)).filter(t -> t);

        val info = new ResourcesSchedule.ResourceInfo(resource.getId(),
                ResourceRoutines.getNameI18n(resource, lang).getOrElse(""),
                ResourceRoutines.getAlterNameI18n(resource, lang).getOrElse(""),
                ResourceRoutines.getResourceEmail(resource), resource.getType(),
                resource.getPhone().isPresent(), resource.getVideo().isPresent(),
                resource.getFloorNum(), ResourceRoutines.getGroupNameI18n(resource, lang),
                canAdmin,
                Option.when(!isBookableOnly, canBook),
                ResourceRoutines.getProtectionMessageI18n(resource, lang).filterNot(x -> canBook),
                resource.getMapUrl());

        // If the user can admin, he has no restrictions
        return new ResourcesSchedule.Resource(info,
                restrictions
                        .map(r -> ResourcesSchedule.RestrictionInterval.cons(r, resourceTz, tz, lang))
                        .filterNot(r -> canAdmin.getOrElse(false)),
                events);
    }

    public static OfficesInfo.OfficeInfo toOfficeInfo(Office office, DateTimeZone currentTz, Language lang) {
        val officeTz = OfficeManager.getOfficeTimeZone(office);
        val tzOffset = officeTz.getOffset(Instant.now()) - currentTz.getOffset(Instant.now());

        return new OfficesInfo.OfficeInfo(
                ResourceRoutines.getNameI18n(office, lang), office.getId(), getCity(office, lang), tzOffset);
    }

    private static boolean isOther(Office office) {
        return office.getName().startsWith("• ");
    }

    private static String getCity(Office office, Language lang) {
        if (office.getName().startsWith("Датацентр")) {
            return new NameI18n("Датацентры", "Datacenters").getName(lang);
        }
        if (isOther(office)) {
            return new NameI18n("Прочее", "Other").getName(lang);
        }
        return ResourceRoutines.getCityNameI18n(office, lang).getOrElse("");
    }

    private static Comparator<Office> officesComparator(Option<Office> userOffice, Language lang) {
        val userCity = userOffice.map(o -> getCity(o, Language.RUSSIAN));

        val preferredCities = Cf.list("Москва", "Санкт-Петербург");
        val otherCities = Cf.list("Датацентры", "Прочее");

        Comparator<Office> preferredCityComparator = Comparator.naturalComparatorBy(o -> {
            val city = getCity(o, Language.RUSSIAN);

            return userCity.isSome(city) ? -1
                    : preferredCities.containsTs(city) ? preferredCities.indexOfTs(city)
                    : otherCities.containsTs(city) ? preferredCities.size() + 1 + otherCities.indexOfTs(city)
                    : preferredCities.size();
        });

        Comparator<Office> preferredOfficeComparator = Comparator.naturalComparatorBy(
                o -> userOffice.isMatch(u -> u.getId().equals(o.getId())) ? -1
                        : preferredCities.containsTs(getCity(o, Language.RUSSIAN)) ? o.getId() : 0);

        return preferredCityComparator
                .thenComparing(Comparator.naturalComparatorBy(o -> getCity(o, lang)))
                .thenComparing(preferredOfficeComparator)
                .thenComparing(Comparator.naturalComparatorBy(o -> ResourceRoutines.getNameI18n(o, lang)));
    }

    public ListF<ResourceInfo> sortedByOffices(
            ListF<ResourceInfo> resources, Option<Office> userOffice, Language lang) {
        return resources.size() < 2 ? resources
                : resources.sorted(officesComparator(userOffice, lang).compose(ResourceInfo::getOffice));
    }

    static ListF<Office> sortOfficesInPreferredOrder(ListF<Office> offices, Option<Office> userOffice, Language lang) {
        val officesByCity = Cf2.stableGroupBy(
                offices.sorted(officesComparator(userOffice, lang)), o -> getCity(o, lang)).values();

        return officesByCity.map(ListF::first).plus(officesByCity.flatMap(s -> s.drop(1)));
    }

    private ActionInfo getActionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }

    private ListF<Long> internalOfficeIdsByStaffIdsInSameOrder(ListF<Long> officeStaffIds) {
        ListF<Office> offices = officeManager.getOfficesByStaffIds(officeStaffIds);

        return officeStaffIds.map(id -> {
            Option<Office> office = offices.find(o -> o.getStaffId().get().equals(id));
            return office.get().getId();
        });
    }
}
