package ru.yandex.calendar.frontend.webNew;

import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.webNew.dto.in.UserSettingsData;
import ru.yandex.calendar.frontend.webNew.dto.in.UserTimezoneData;
import ru.yandex.calendar.frontend.webNew.dto.out.ContactsInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.NowTimeInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.UserSettingsInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.WebUserInfo;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.contact.ContactRoutines;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.EventWithRelations;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantId;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.UserParticipantInfo;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.LockTransactionManager;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DateTimeManager;
import ru.yandex.commune.a3.action.parameter.IllegalParameterException;
import ru.yandex.inside.passport.AbstractPassportUid;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.time.InstantInterval;

public class WebNewUserManager {

    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private UserManager userManager;
    @Autowired
    private OfficeManager officeManager;
    @Autowired
    private DateTimeManager dateTimeManager;
    @Autowired
    private LockTransactionManager lockTransactionManager;
    @Autowired
    private EventRoutines eventRoutines;
    @Autowired
    private ContactRoutines contactRoutines;

    public UserSettingsInfo getUserSettings(PassportUid uid) {
        SettingsInfo settings = settingsRoutines.getSettingsByUid(uid);
        UserInfo userInfo = userManager.getUserInfo(uid);

        Option<Long> activeOfficeId = settings.getYt().filterMap(SettingsYt.getActiveOfficeIdF());
        Option<Office> currentOffice = officeManager.getOfficesByIds(activeOfficeId).singleO();

        return new UserSettingsInfo(settings, userInfo, currentOffice);
    }

    public NowTimeInfo getNowTime(Option<PassportUid> uid, Option<DateTimeZone> tzO) {
        if (uid.isPresent()) {
            Option<SettingsInfo> settingsInfo = settingsRoutines.getSettingsByUidIfExists(uid.get());
            Option<Settings> settings = settingsInfo.map(SettingsInfo.getCommonF());
            Option<SettingsYt> settingsYt = settingsInfo.filterMap(SettingsInfo.getYtF());

            Option<Long> activeOfficeId = settingsYt.filterMap(SettingsYt.getActiveOfficeIdF());
            Option<Office> currentOffice = officeManager.getOfficesByIds(activeOfficeId).singleO();

            Function<String, DateTimeZone> tzF = AuxDateTime.getVerifyDateTimeZoneF();
            DateTimeZone defaultTz = dateTimeManager.getDefaultTimezone();

            DateTimeZone tz = settings.map(Settings.getTimezoneJavaidF().andThen(tzF)).getOrElse(defaultTz);
            String geoTzId = settings.map(Settings.getGeoTzJavaidF()).getOrElse(defaultTz.getID());
            Option<String> officeTz = currentOffice.filterMap(Office.getTimezoneIdF());

            return new NowTimeInfo(getNow().toDateTime(tz).toLocalDateTime(), tz.getID(), Option.of(geoTzId), officeTz);

        } else if (tzO.isPresent()) {
            DateTimeZone tz = tzO.get();

            return new NowTimeInfo(getNow().toDateTime(tz).toLocalDateTime(), tz.getID(), Option.empty(), Option.empty());

        } else {
            throw new IllegalParameterException("tz", "No timezone source specified");
        }
    }

    public void updateUserSettings(final PassportUid uid, final UserSettingsData data) {
        DateTimeZone userTz = dateTimeManager.getTimeZoneForUid(uid);

        lockTransactionManager.lockAndDoInTransaction(LockResource.settingsUser(uid), () -> {
            settingsRoutines.updateOrCreateSettingsByUid(data.getSettings(userTz), uid);
            settingsRoutines.updateSettingsYtByUid(data.getSettingsYt(), uid);
            settingsRoutines.updateTimezones(uid, data.getTz(), data.getLastOfferedGeoTz());
        });
    }

    public void updateUserTimezone(final PassportUid uid, final UserTimezoneData data) {
        lockTransactionManager.lockAndDoInTransaction(LockResource.settingsUser(uid),
                () -> settingsRoutines.updateTimezones(
                        uid,
                        Option.of(data.getTz()),
                        Option.of(data.getLastOfferedGeoTz())));
    }

    // CAL-5787
    public ContactsInfo findFavoriteContacts(PassportUid uid, int limit, Language lang) {
        val now = getNow();
        val interval = new InstantInterval(now.minus(Duration.standardDays(60)), now);

        val recentUserAttendedEvents = eventRoutines.findLastUserAttendedEvents(uid, 30, interval);
        val recentEventsParticipants = getRecentEventsParticipants(recentUserAttendedEvents);
        val topParticipants = getTopParticipants(uid, limit, recentEventsParticipants);

        val emails = topParticipants.keySet();
        val nameByEmail = contactRoutines.getI18NamesByEmails(uid, Cf.toList(emails)).toMap();

        return new ContactsInfo(emails.stream()
                .map(email -> {
                    val name = nameByEmail.getOrThrow(email).map(NameI18n.getNameF(lang)).getOrElse("");
                    val login = topParticipants.computeIfAbsent(email, x -> {
                        throw new IllegalStateException();
                    });
                    return new ContactsInfo.ContactInfo(name, email, login);
                })
                .collect(CollectorsF.toList()));
    }

    static LinkedHashMap<Email, Option<String>> getTopParticipants(PassportUid uid,
                                                                   int limit,
                                                                   Stream<UserParticipantInfo> recentEventsParticipants) {
        return StreamEx.of(recentEventsParticipants)
                .remove(p -> p.getDecision() == Decision.NO)
                .remove(p -> p.getUid().isSome(uid))
                .remove(p -> p.getYtSettingsIfYtUser().exists(SettingsYt::getIsDismissed))
                .sortedBy(x -> x.getEmail().getEmail())
                .groupRuns((left, right) -> left.getEmail().equals(right.getEmail()))
                .reverseSorted(Comparator.comparingInt(List::size))
                .limit(limit)
                .map(infos -> infos.stream().reduce((left, right) -> left.isYandexUser() ? left : right))
                .flatMap(Optional::stream)
                .mapToEntry(UserParticipantInfo::getEmail, UserParticipantInfo::getLoginIfYandexUser)
                .toCustomMap(LinkedHashMap::new);
    }

    private static StreamEx<UserParticipantInfo> getRecentEventsParticipants(Collection<EventWithRelations> recentUserAttendedEvents) {
        return StreamEx.of(recentUserAttendedEvents)
                .map(EventWithRelations::getParticipants)
                .flatCollection(Participants::getUserParticipantsSafeWithInconsistent);
    }

    public WebUserInfo getWebUserInfoByParticipantId(PassportUid uid, ParticipantId id, Language lang) {
        return getWebUserInfos(uid, Either.left(Cf.list(id)), lang).single().get2();
    }

    public Map<ParticipantId, WebUserInfo> getWebUserInfos(PassportUid uid, Collection<ParticipantId> participants, Language lang) {
        return getWebUserInfos(uid, Either.left(Cf.toList(participants)), lang).toMap();
    }

    public Tuple2List<ParticipantId, WebUserInfo> getWebUserInfos(PassportUid uid,
                                                                  Either<ListF<ParticipantId>, CollectionF<? extends UserParticipantInfo>> participants, Language lang) {
        Validate.forAll(participants.fold(is -> is, ps -> Cf.list()), ParticipantId.isResourceF().notF());

        ListF<ParticipantId> ids = participants.fold(is -> is, ps -> ps.map(ParticipantInfo::getId));

        ListF<PassportUid> uids = ids.filterMap(ParticipantId.getUidIfYandexUserF());
        ListF<Email> externalEmails = ids.filterMap(ParticipantId.getEmailIfExternalUserF());

        MapF<PassportUid, SettingsInfo> settingsByUid = participants.fold(
                is -> settingsRoutines.getSettingsByUidBatch(uids),
                ps -> ps.filterMap(UserParticipantInfo.getSettingsIfUserF()).toMapMappingToKey(SettingsInfo::getUid));

        ListF<Email> settingsEmails = settingsByUid.values().map(SettingsInfo.getCommonF().andThen(Settings.getEmailF()));

        final MapF<Email, Option<NameI18n>> nameByEmail = contactRoutines.getI18NamesByEmails(
                uid, externalEmails.plus(settingsEmails)).toMap();

        return ids.zipWith(id -> {
            {
                if (id.isYandexUser()) {
                    SettingsInfo settingsInfo = settingsByUid.getOrThrow(id.getUid());
                    Email email = settingsInfo.getEmail();
                    String name = nameByEmail.getOrThrow(email).map(NameI18n.getNameF(lang)).getOrElse("");
                    Option<String> login = settingsByUid.getOrThrow(id.getUid()).getUserLogin();
                    Option<Long> officeId = settingsByUid.getOrThrow(id.getUid()).getYt().filterMap(
                            SettingsYt.getActiveOfficeIdF());

                    return new WebUserInfo(Option.of(settingsInfo.getUid().getUid()), name, email, login, officeId);

                } else {
                    Option<Long> yandexUid = id.getUidIfYandexUser().map(AbstractPassportUid::getUid);
                    Email email = id.getInviteeEmailForExternalUser();
                    String name = nameByEmail.getOrThrow(email).map(NameI18n.getNameF(lang)).getOrElse("");

                    return new WebUserInfo(yandexUid, name, email, Option.<String>empty(), Option.<Long>empty());
                }
            }
        });
    }

    private Instant getNow() {
        return getActionInfo().getNow();
    }

    private ActionInfo getActionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }
}
