package ru.yandex.calendar.frontend.webNew.actions;

import java.io.IOException;
import java.io.StringWriter;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.bind.BindJson;
import ru.yandex.calendar.frontend.api.ApiBender;
import ru.yandex.calendar.frontend.bender.FilterablePojo;
import ru.yandex.calendar.frontend.bender.FilteringFields;
import ru.yandex.calendar.frontend.bender.WebDate;
import ru.yandex.calendar.frontend.webNew.WebNewAvailabilityManager;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailDisplayMode;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailParameters;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailShapeType;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailabilitiesData;
import ru.yandex.calendar.frontend.webNew.dto.in.EmailsData;
import ru.yandex.calendar.frontend.webNew.dto.in.SuggestData;
import ru.yandex.calendar.frontend.webNew.dto.out.HolidaysInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ReservationInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ResourcesSchedule;
import ru.yandex.calendar.frontend.webNew.dto.out.StatusResult;
import ru.yandex.calendar.frontend.webNew.dto.out.SubjectsAvailabilities;
import ru.yandex.calendar.frontend.webNew.dto.out.SubjectsAvailabilityIntervals;
import ru.yandex.calendar.frontend.webNew.dto.out.SuggestDatesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.SuggestInfo;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SpecialUsers;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.holidays.OutputMode;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.serialize.BenderJsonGeneratorWrapper;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.email.Email;

@Slf4j
@ActionContainer
public class AvailabilityActions {
    @Autowired
    private WebNewAvailabilityManager webNewAvailabilityManager;

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public FilterablePojo<SubjectsAvailabilityIntervals> getAvailabilityIntervals(
            @RequestParam("uid") PassportUid uid,
            @RequestParam(value = "emails", required = false) ListF<Email> emails,
            @RequestParam("date") Option<LocalDate> date,
            @RequestParam("from") Option<WebDate> from,
            @RequestParam("to") Option<WebDate> till,
            @RequestParam("display") Option<String> display,
            @RequestParam("shape") Option<String> shape,
            @RequestParam("idsOnly") Option<Boolean> idsOnly,
            @AvailParameters.Bind AvailParameters params,
            @BindJson Option<EmailsData> emailsData,
            @RequestParam("lang") Option<Language> lang,
            @RequestParam("fields") Option<FilteringFields> fields) {
        FilterablePojo<SubjectsAvailabilityIntervals> result = FilterablePojo.wrap(
                webNewAvailabilityManager.getAvailabilityIntervals(
                        uid, emails.plus(emailsData.flatMap(EmailsData::getEmails)), date, from, till,
                        AvailDisplayMode.of(display), AvailShapeType.of(shape, idsOnly),
                        params, lang.getOrElse(Language.RUSSIAN)),
                fields);

        logResultIfNeeded(uid, result.getBendable());
        return result;
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public SubjectsAvailabilities getAvailabilities(
            @RequestParam("uid") PassportUid uid,
            @BindJson AvailabilitiesData data,
            @AvailParameters.Bind AvailParameters params) {
        SubjectsAvailabilities result = webNewAvailabilityManager.getAvailabilities(uid, data, params);
        logResultIfNeeded(uid, result);
        return result;
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public SuggestInfo suggestMeetingTimes(
            @RequestParam("uid") PassportUid uid,
            @BindJson SuggestData suggestData,
            @AvailParameters.Bind AvailParameters params,
            @RequestParam("lang") Option<Language> lang) {
        return webNewAvailabilityManager.suggest(uid, suggestData, params, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public ResourcesSchedule suggestMeetingResources(
            @RequestParam("uid") PassportUid uid,
            @BindJson SuggestData suggestData,
            @AvailParameters.Bind AvailParameters params,
            @RequestParam("lang") Option<Language> lang) {
        return webNewAvailabilityManager.suggestResources(uid, suggestData, params, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public SuggestDatesInfo suggestMeetingDates(
            @RequestParam("uid") PassportUid uid,
            @BindJson SuggestData suggestData,
            @AvailParameters.Bind AvailParameters params) {
        return webNewAvailabilityManager.suggestDates(uid, suggestData, params);
    }

    @Action
    public ReservationInfo reserveResources(
            @RequestParam("uid") PassportUid uid,
            @BindJson AvailabilitiesData data,
            @RequestParam("reservationId") long reservationId,
            @RequestParam("keepIfAnyBusy") Option<Boolean> keepIfAnyBusy,
            @RequestParam("keepIfResourceBusy") Option<Email> keepIfResourceBusy,
            @AvailParameters.Bind AvailParameters params) {
        return webNewAvailabilityManager.reserveResources(
                uid, data, reservationId, keepIfAnyBusy, keepIfResourceBusy, params);
    }

    @Action
    public StatusResult cancelResourcesReservation(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("reservationId") long reservationId) {
        webNewAvailabilityManager.cancelResourcesReservation(uid, reservationId);
        return StatusResult.ok();
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public HolidaysInfo getHolidays(
            @RequestParam("uid") Option<PassportUid> uid,
            @RequestParam("from") LocalDate from,
            @RequestParam("to") LocalDate to,
            @RequestParam("for") String holidaysFor,
            @RequestParam("outMode") Option<OutputMode> outMode) {
        return webNewAvailabilityManager.getHolidays(uid, from, to, holidaysFor, outMode);
    }

    private void logResultIfNeeded(PassportUid uid, Object result) {
        // added to investigate causes of CAL-8002, CAL-8226
        if (uid.equalsTs(SpecialUsers.WHISTLER)) {
            BenderMapper mapperForLogging = new BenderMapper(ApiBender.getConfigurationForLogging());

            try {
                StringWriter wr = new StringWriter();
                JsonGenerator gen = new JsonFactory().createGenerator(wr);
                BenderJsonWriter jw = new BenderJsonGeneratorWrapper(gen);
                mapperForLogging.serializeJson(result, jw);
                jw.close();
                log.debug("Availability Investigation: {}", wr);
            } catch (IOException ignore) {
            }
        }
    }
}
