package ru.yandex.calendar.frontend.webNew.actions;

import java.util.Optional;

import lombok.val;
import one.util.streamex.StreamEx;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.a3.bind.BindJson;
import ru.yandex.calendar.frontend.a3.converters.ConverterToInstantMillis;
import ru.yandex.calendar.frontend.a3.interceptors.WithTlTimeoutByProduct;
import ru.yandex.calendar.frontend.bender.FilterablePojo;
import ru.yandex.calendar.frontend.bender.FilteringFields;
import ru.yandex.calendar.frontend.bender.WebDate;
import ru.yandex.calendar.frontend.webNew.WebNewEventManager;
import ru.yandex.calendar.frontend.webNew.dto.in.ExternalIdsData;
import ru.yandex.calendar.frontend.webNew.dto.in.RepetitionWithStartData;
import ru.yandex.calendar.frontend.webNew.dto.in.ReplyData;
import ru.yandex.calendar.frontend.webNew.dto.in.WebEventData;
import ru.yandex.calendar.frontend.webNew.dto.in.WebEventUserData;
import ru.yandex.calendar.frontend.webNew.dto.out.EventInfoShort;
import ru.yandex.calendar.frontend.webNew.dto.out.EventInfoWithSeries;
import ru.yandex.calendar.frontend.webNew.dto.out.EventsBrief;
import ru.yandex.calendar.frontend.webNew.dto.out.EventsCountInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.EventsInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ModifiedEventIdsInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ModifyEventResult;
import ru.yandex.calendar.frontend.webNew.dto.out.MoveResourceEventsIds;
import ru.yandex.calendar.frontend.webNew.dto.out.ParseTimeInEventNameResult;
import ru.yandex.calendar.frontend.webNew.dto.out.RepetitionDescription;
import ru.yandex.calendar.frontend.webNew.dto.out.StatusResult;
import ru.yandex.calendar.frontend.webNew.dto.out.WebEventInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.WebUserParticipantsInfo;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.web.IdOrExternalId;
import ru.yandex.calendar.logic.update.DistributedSemaphore;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestListParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;

@ActionContainer
public class EventActions {

    @Autowired
    private WebNewEventManager webNewEventManager;
    @Autowired
    private DistributedSemaphore distributedSemaphore;

    private static ActionInfo getActionInfo() {
        return CalendarRequest.getCurrent().getActionInfo();
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @WithTlTimeoutByProduct(pub = 29 * 1000, yt = 29 * 10000)
    @Action
    public ModifyEventResult createEvent(@RequestParam("uid") PassportUid uid,
                                         @RequestParam("externalId") Option<String> extId,
                                         @RequestParam("tz") Option<DateTimeZone> tz,
                                         @BindJson WebEventData createEventData) {
        return webNewEventManager.createEvent(uid, extId, tz, createEventData, getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @WithTlTimeoutByProduct(pub = 29 * 1000, yt = 29 * 10000)
    @Action
    public ModifyEventResult updateEvent(@RequestParam("uid") PassportUid uid,
                                         @RequestParam("id") Option<Long> eventId,
                                         @RequestParam("layerId") Option<Long> layerId,
                                         @RequestParam("externalId") Option<String> eventExtId,
                                         @RequestParam("privateToken") Option<String> privateToken,
                                         @RequestParam("sequence") Option<Integer> sequence,
                                         @RequestParam("tz") Option<DateTimeZone> tz,
                                         @BindJson WebEventData webEventData,
                                         @RequestParam("mailToAll") Option<Boolean> mailToAll,
                                         @RequestParam("applyToFuture") Option<Boolean> applyToFuture,
                                         @RequestParam("instanceStartTs") Option<LocalDateTime> instanceStartTs) {
        return webNewEventManager.updateEvent(uid, IdOrExternalId.fromOptions(eventId, eventExtId),
                layerId, privateToken, sequence, tz, webEventData, applyToFuture, instanceStartTs, mailToAll, getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult attachEvent(@RequestParam("uid") PassportUid uid,
                                    @RequestParam("id") long eventId,
                                    @BindJson Option<WebEventUserData> data) {
        return webNewEventManager.attachEvent(uid, eventId, data.getOrElse(WebEventUserData::empty), getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @WithTlTimeoutByProduct(pub = 29 * 1000, yt = 29 * 10000)
    @Action
    public StatusResult deleteEvent(@RequestParam("uid") PassportUid uid,
                                    @RequestParam("id") Option<Long> eventId,
                                    @RequestParam("externalId") Option<String> eventExtId,
                                    @RequestParam("sequence") Option<Integer> sequence,
                                    @RequestParam("tz") Option<DateTimeZone> tz,
                                    @RequestParam("applyToFuture") Option<Boolean> applyToFuture,
                                    @RequestParam("instanceStartTs") Option<LocalDateTime> instanceStartTs) {
        return webNewEventManager.deleteEvent(uid, IdOrExternalId.fromOptions(eventId, eventExtId),
                sequence, applyToFuture.getOrElse(Boolean.FALSE), instanceStartTs, getActionInfo());
    }

    @WithTlTimeoutByProduct(yt = 25000, pub = 10000)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult detachEvent(@RequestParam("uid") PassportUid uid,
                                    @RequestParam("id") long eventId,
                                    @RequestParam("layerId") Option<Long> layerId) {
        return webNewEventManager.detachEvent(uid, eventId, layerId, getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult confirmEventRepetition(@RequestParam("uid") PassportUid uid,
                                               @RequestParam("id") long eventId) {
        return webNewEventManager.confirmEventRepetition(eventId, getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult deleteFutureEvents(@RequestParam("uid") PassportUid uid,
                                           @RequestParam("id") long eventId) {
        return webNewEventManager.deleteFutureEvents(uid, eventId, getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public WebUserParticipantsInfo getAttendees(@RequestParam("uid") PassportUid uid,
                                                @RequestParam("eventId") long eventId,
                                                @RequestParam("lang") Option<Language> lang) {
        val event = webNewEventManager.getEvent(uid, IdOrExternalId.id(eventId), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), lang.getOrElse(Language.RUSSIAN),
                Optional.empty(), getActionInfo());
        return new WebUserParticipantsInfo(event.getAttendees());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public WebUserParticipantsInfo getOptionalAttendees(@RequestParam("uid") PassportUid uid,
                                                @RequestParam("eventId") long eventId,
                                                @RequestParam("lang") Option<Language> lang) {
        val event = webNewEventManager.getEvent(uid, IdOrExternalId.id(eventId), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), lang.getOrElse(Language.RUSSIAN),
                Optional.empty(), getActionInfo());
        return new WebUserParticipantsInfo(event.getOptionalAttendees());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public WebEventInfo getEvent(@RequestParam("uid") PassportUid uid,
                                 @RequestParam("eventId") Option<Long> eventId,
                                 @RequestParam("layerId") Option<Long> layerId,
                                 @RequestParam("externalId") Option<String> eventExtId,
                                 @RequestParam("privateToken") Option<String> privateToken,
                                 @RequestParam("tz") Option<DateTimeZone> tz,
                                 @RequestParam("instanceStartTs") Option<LocalDateTime> instanceStartTs,
                                 @RequestParam("recurrenceAsOccurrence") Option<Boolean> recurrenceAsOccurrence,
                                 @RequestParam("linksign") Option<String> linkSignKey,
                                 @RequestParam("lang") Option<Language> lang,
                                 @RequestParam("participantUid") Option<PassportUid> participantUid) {
        return webNewEventManager.getEvent(uid, IdOrExternalId.fromOptions(eventId, eventExtId),
                layerId, privateToken, tz, instanceStartTs, recurrenceAsOccurrence,
                linkSignKey, lang.getOrElse(Language.RUSSIAN), participantUid.toOptional(), getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public EventInfoWithSeries getEventWithSeries(@RequestParam("uid") PassportUid uid,
                                                  @RequestParam("externalId") String externalId,
                                                  @RequestParam("lang") Option<Language> lang) {
        return webNewEventManager.getEvent(uid, externalId, lang.getOrElse(Language.RUSSIAN), getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public EventsInfo getUserEvents(@RequestParam("uid") Option<PassportUid> uid,
                                    @RequestParam("actorUid") Option<PassportUid> actorUid,
                                    @RequestParam("targetUid") Option<PassportUid> targetUid,
                                    @RequestParam("from") WebDate from,
                                    @RequestParam("to") WebDate till,
                                    @RequestParam("linksign") Option<String> linkSignKey,
                                    @RequestParam("limitAttendees") Option<Boolean> limitAttendees,
                                    @RequestParam("limitSize") Option<Integer> limitSize
    ) {
        return webNewEventManager.getEvents(
                Optional.of(uid.orElse(() -> actorUid)
                               .getOrThrow(() -> new IllegalArgumentException("uid or actorUid parameter required"))),
                targetUid.toOptional(), Cf.list(), Option.empty(), Option.empty(),
                from, Option.of(till), Option.of(true), Option.of(true),
                Option.of(false), Option.of(true),
                Optional.empty(), Option.empty(), linkSignKey, limitAttendees.getOrElse(Boolean.FALSE), getActionInfo(), limitSize);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public MoveResourceEventsIds moveResource(@RequestParam("uid") PassportUid uid,
                                              @RequestParam("sourceEventUid") PassportUid sourceEventUid,
                                              @RequestParam("sourceId") long sourceEventId,
                                              @RequestParam("targetId") long targetEventId,
                                              @RequestParam("resource") String resourceEmail,
                                              @RequestParam("targetEventUid") Option<PassportUid> targetEventUid,
                                              @RequestParam("instanceStartTs") Option<LocalDateTime> instanceStartTs) {
        return webNewEventManager.moveResource(uid, sourceEventUid, sourceEventId, targetEventId, resourceEmail,
                targetEventUid.toOptional(), instanceStartTs.toOptional(), getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public FilterablePojo<EventsInfo> getEvents(@RequestParam("uid") Option<PassportUid> uid,
                                                @RequestParam("actorUid") Option<PassportUid> actorUid,
                                                @RequestParam("targetUid") Option<PassportUid> targetUid,
                                                @RequestParam(value = "layerId", required = false) ListF<String> layerIds,
                                                @RequestParam(value = "layerToken") Option<String> layerToken,
                                                @RequestParam(value = "layerToggledOn") Option<Boolean> layerToggledOn,
                                                @RequestParam("from") Option<WebDate> from,
                                                @RequestParam("to") Option<WebDate> till,
                                                @RequestParam("eventId") Option<Long> eventIdO,
                                                @RequestParam(value = "externalId", required = false) ListF<String> externalIds,
                                                @BindJson Option<ExternalIdsData> externalIdsData,
                                                @RequestParam("mergeLayers") Option<Boolean> mergeLayers,
                                                @RequestParam("opaqueOnly") Option<Boolean> opaqueOnly,
                                                @RequestParam("showDeclined") Option<Boolean> showDeclined,
                                                @RequestParam("hideAbsences") Option<Boolean> hideAbsences,
                                                @RequestParam("tz") Option<DateTimeZone> tz,
                                                @RequestParam("linksign") Option<String> linkSignKey,
                                                @RequestParam("lang") Option<Language> lang,
                                                @RequestParam("fields") Option<FilteringFields> fields,
                                                @RequestParam("limitAttendees") Option<Boolean> limitAttendees,
                                                @RequestParam("limitSize") Option<Integer> limitSize
                                                ) {
        val allExternalIds = StreamEx.of(externalIds)
                .append(StreamEx.of(externalIdsData).flatCollection(ExternalIdsData::getExternalIds))
                .collect(CollectorsF.toList());
        if (from.isEmpty() && tz.isPresent()) {
            from = Option.of(WebDate.localDate(LocalDate.now().toDateTimeAtStartOfDay(tz.get()).toLocalDate()));
        }
        EventsInfo firstPartOfEvents = webNewEventManager.getEvents(
                uid.orElse(() -> actorUid).toOptional(),
                targetUid.toOptional(),
                layerIds,
                layerToken,
                layerToggledOn,
                from.get(),
                till,
                mergeLayers,
                opaqueOnly,
                showDeclined,
                hideAbsences,
                tz.toOptional(),
                lang,
                linkSignKey,
                eventIdO,
                allExternalIds,
                limitAttendees.getOrElse(Boolean.FALSE),
                getActionInfo(),
                limitSize);
        if (till.isEmpty() && limitSize.isPresent()) {
            // according to https://st.yandex-team.ru/GREG-956#5f9993af113a685370d3a377
            // adding events to end of day when last event with limitSize
            Option<WebEventInfo> webEventInfos = firstPartOfEvents.getEvents().lastO();
            if (webEventInfos.isPresent()) {
                WebDate newTill = WebDate.localDate(webEventInfos.get().getEndTs().toLocalDateTime().plusDays(1).toLocalDate().toDateTimeAtStartOfDay().toLocalDate());
                EventsInfo secondPartOfEvents = webNewEventManager.getEvents(
                        uid.orElse(() -> actorUid).toOptional(), targetUid.toOptional(), layerIds, layerToken, layerToggledOn,
                        from.get(), Option.of(newTill), mergeLayers, opaqueOnly, showDeclined, hideAbsences, tz.toOptional(), lang,
                        linkSignKey, eventIdO, allExternalIds, limitAttendees.getOrElse(Boolean.FALSE),
                        getActionInfo(), Option.empty());
                return FilterablePojo.wrap(secondPartOfEvents, fields);
            }
        }
        return FilterablePojo.wrap(firstPartOfEvents, fields);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public FilterablePojo<EventsInfo> getModifiedEvents(@RequestParam("uid") Option<PassportUid> uid,
                                                        @RequestParam("actorUid") Option<PassportUid> actorUid,
                                                        @RequestParam("targetUid") Option<PassportUid> targetUid,
                                                        @RequestParam(value = "since", customConverter = ConverterToInstantMillis.class) Option<Instant> since,
                                                        @RequestParam(value = "layerId", required = false) ListF<String> layerIds,
                                                        @RequestParam(value = "layerToggledOn") Option<Boolean> layerToggledOn,
                                                        @RequestParam("tz") Option<DateTimeZone> tz,
                                                        @RequestParam("linksign") Option<String> linkSignKey,
                                                        @RequestParam("lang") Option<Language> lang,
                                                        @RequestParam("fields") Option<FilteringFields> fields) {
        return FilterablePojo.wrap(
                webNewEventManager.getModifiedEvents(
                        uid.orElse(() -> actorUid).getOrThrow(() -> new IllegalArgumentException("uid or actorUid parameter required")),
                        targetUid.toOptional(), layerIds, layerToggledOn, since, tz, lang,
                        linkSignKey, getActionInfo()),
                fields);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public ModifiedEventIdsInfo getModifiedEventIds(@RequestParam("uid") PassportUid uid,
                                                    @RequestParam(value = "since", customConverter = ConverterToInstantMillis.class) Instant since,
                                                    @RequestParam(value = "layerId", required = false) ListF<String> layerIds,
                                                    @RequestParam(value = "layerToggledOn") Option<Boolean> layerToggledOn) {
        try (var lock = distributedSemaphore.forceTryAcquire("getModifiedEventIds")) {
            return webNewEventManager.getModifiedEventIds(uid, since, layerIds, layerToggledOn);
        }
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public EventsBrief getEventsBrief(@RequestParam("uid") PassportUid uid,
                                      @RequestListParam("eventIds") ListF<Long> eventIds,
                                      @RequestParam("forResource") Option<Boolean> forResource,
                                      @RequestParam("lang") Option<Language> lang) {
        return webNewEventManager.getEventsBrief(uid, eventIds, forResource.getOrElse(Boolean.FALSE), lang.getOrElse(Language.RUSSIAN),
                getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public EventsCountInfo countEvents(@RequestParam("uid") Option<PassportUid> uid,
                                       @RequestParam("actorUid") Option<PassportUid> actorUid,
                                       @RequestParam("targetUid") Option<PassportUid> targetUid,
                                       @RequestParam(value = "layerId", required = false) ListF<Long> layerIds,
                                       @RequestParam(value = "layerToken") Option<String> layerToken,
                                       @RequestParam(value = "layerToggledOn") Option<Boolean> layerToggledOn,
                                       @RequestParam("from") LocalDate from,
                                       @RequestParam("to") LocalDate till,
                                       @RequestParam("opaqueOnly") Option<Boolean> opaqueOnly,
                                       @RequestParam("showDeclined") Option<Boolean> showDeclined,
                                       @RequestParam("hideAbsences") Option<Boolean> hideAbsences,
                                       @RequestParam("tz") Option<DateTimeZone> tz) {
        return webNewEventManager.countEvents(uid.orElse(() -> actorUid).toOptional(), targetUid.toOptional(), layerIds,
                layerToken, layerToggledOn, from, till, opaqueOnly, showDeclined, hideAbsences,
                tz.toOptional(), getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public EventInfoShort handleReply(@RequestParam("uid") PassportUid uid,
                                      @BindJson ReplyData replyData) {
        return webNewEventManager.handleReply(uid, replyData, getActionInfo());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public RepetitionDescription getRepetitionDescription(@RequestParam("uid") PassportUid uid,
                                                          @RequestParam("tz") Option<DateTimeZone> tz,
                                                          @RequestParam("lang") Option<Language> lang,
                                                          @BindJson RepetitionWithStartData repetitionData) {
        return webNewEventManager.getRepetitionDescription(uid, repetitionData, tz, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.CUT_OFF)
    @Action
    public ParseTimeInEventNameResult parseTimeInEventName(@RequestParam("uid") PassportUid uid,
                                                           @RequestParam("name") String name,
                                                           @RequestParam("date") LocalDate date) {
        return webNewEventManager.parseTimeInEventName(name, date);
    }
}
