package ru.yandex.calendar.frontend.webNew.actions;

import lombok.val;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.bind.BindJson;
import ru.yandex.calendar.frontend.web.cmd.run.CommandRunException;
import ru.yandex.calendar.frontend.web.cmd.run.Situation;
import ru.yandex.calendar.frontend.webNew.WebNewLayerManager;
import ru.yandex.calendar.frontend.webNew.bind.ImportIcsDataBinder;
import ru.yandex.calendar.frontend.webNew.dto.in.ImportIcsData;
import ru.yandex.calendar.frontend.webNew.dto.in.LayerData;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerIdInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerOwnerInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.LayerPrivateToken;
import ru.yandex.calendar.frontend.webNew.dto.out.LayersInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.StatusResult;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.thread.WithTlTimeoutInMillis;

@ActionContainer
public class LayerActions {
    @Autowired
    private WebNewLayerManager webNewLayerManager;

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public LayerIdInfo createLayer(
            @RequestParam("uid") PassportUid uid,
            @BindJson LayerData data)
    {
        return webNewLayerManager.createLayer(uid, data);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult updateLayer(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("id") long layerId,
            @RequestParam("applyNotificationsToEvents") boolean applyToEvents,
            @BindJson LayerData data)
    {
        return webNewLayerManager.updateLayer(uid, layerId, applyToEvents, data);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult toggleLayer(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("id") long layerId,
            @RequestParam("on") boolean on)
    {
        return webNewLayerManager.toggleLayer(uid, layerId, on);
    }

    @WithTlTimeoutInMillis(29000)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult deleteLayer(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("id") long layerId,
            @RequestParam("recipientLayerId") Option<Long> recipientLayerId,
            @RequestParam("newDefaultLayerId") Option<Long> newDefaultLayerId)
    {
        return webNewLayerManager.deleteLayer(uid, layerId, recipientLayerId, newDefaultLayerId);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public LayerInfo getLayer(
            @RequestParam("uid") Option<PassportUid> uid,
            @RequestParam("id") Option<Long> layerId,
            @RequestParam("lang") Option<Language> lang,
            @RequestParam("layerToken") Option<String> layerToken)
    {
        if (layerToken.isPresent()) {
            val foundAndNot = webNewLayerManager.findLayersCanList(
                    Option.empty(), Cf.list(), layerToken, Option.of(true), true, false);
            ListF<Layer> foundLayers = foundAndNot.get1();
            if (foundLayers.isEmpty()) {
                if (layerId.isPresent()) {
                    throw CommandRunException.createSituationF(
                            "layer not found by id " + layerId + " and uid " + uid,
                            Situation.NO_PERMISSIONS_FOR_LAYER_ACTION).apply();
                } else {
                    throw CommandRunException.createSituationF(
                            "layer not found by token " + layerToken + " and uid " + uid,
                            Situation.NO_PERMISSIONS_FOR_LAYER_ACTION).apply();
                }
            }
            Layer layer = foundLayers.get(0);
            PassportUid creatorUid = layer.getCreatorUid();
            Long finalLayerId = layerId.isPresent() ? layerId.get() : layer.getId();
            return webNewLayerManager.getLayer(creatorUid, finalLayerId, lang.getOrElse(Language.RUSSIAN),
                    Option.of(layerUserWithRelations -> layerUserWithRelations.getLayerId() == finalLayerId));
        }
        if (!uid.isPresent()) {
            throw CommandRunException.createSituationF(
                    "provide uid if layerToken is not provided",
                    Situation.MISSING_UID).apply();
        }
        if (!layerId.isPresent()) {
            throw CommandRunException.createSituationF(
                    "provide layerId if layerToken is not provided",
                    Situation.MISSING_LAYER_ID).apply();
        }
        return webNewLayerManager.getLayer(uid.get(), layerId.get(), lang.getOrElse(Language.RUSSIAN), Option.empty());
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public LayerOwnerInfo getOwnerByLayerId(@RequestParam("id") long layerId) {
        return webNewLayerManager.getOwnerByLayerId(layerId);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public LayersInfo getUserLayers(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewLayerManager.getUserLayers(uid, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public LayerPrivateToken obtainLayerPrivateToken(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("id") long layerId,
            @RequestParam("forceNew") Option<Boolean> forceNew)
    {
        return webNewLayerManager.obtainLayerPrivateToken(uid, layerId, forceNew.getOrElse(false));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult handleLayerReply(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("privateToken") String privateToken,
            @RequestParam("decision") Decision decision,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewLayerManager.handleLayerReply(uid, privateToken, decision);
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public LayerInfo shareLayer(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("id") Option<Long> id,
            @RequestParam("privateToken") Option<String> privateToken,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewLayerManager.shareLayer(uid, id, privateToken, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public LayerInfo createIcsFeed(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("url") String url,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewLayerManager.createIcsFeed(uid, url, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public StatusResult forceFeedUpdate(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("id") long feedId)
    {
        return webNewLayerManager.forceFeedUpdate(uid, feedId);
    }

    @WithTlTimeoutInMillis(29000)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    @Action
    public LayerInfo importIcs(
            @RequestParam("uid") PassportUid uid,
            @BindWith(ImportIcsDataBinder.class) ImportIcsData data)
    {
        return webNewLayerManager.importIcs(uid, data);
    }
}
