package ru.yandex.calendar.frontend.webNew.actions;

import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.a3.bind.BindJson;
import ru.yandex.calendar.frontend.bender.WebDate;
import ru.yandex.calendar.frontend.webNew.WebNewResourcesManager;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailParameters;
import ru.yandex.calendar.frontend.webNew.dto.in.FindUsersAndResourcesData;
import ru.yandex.calendar.frontend.webNew.dto.in.IntervalAndRepetitionData;
import ru.yandex.calendar.frontend.webNew.dto.out.OfficesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.OfficesTzInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ResourcesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.ResourcesSchedule;
import ru.yandex.calendar.frontend.webNew.dto.out.UserOrResourceInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.UsersAndResourcesInfo;
import ru.yandex.calendar.frontend.webNew.dto.out.WebResourceInfo;
import ru.yandex.calendar.logic.resource.MultiofficeResourceFilter;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.commune.a3.action.Action;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.email.Email;

@ActionContainer
public class ResourcesActions {
    @Autowired
    private WebNewResourcesManager webNewResourcesManager;

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public ResourcesInfo findResources(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("officeId") Option<Long> officeId,
            @RequestParam("filter") Option<String> filter,
            @RequestParam("prefix") Option<String> resourceNamePrefix,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewResourcesManager.findResources(
                uid, new MultiofficeResourceFilter(officeId, filter, resourceNamePrefix, Cf.list()),
                lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public ResourcesInfo findResourcesByStaffId(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("officeId") Option<Long> officeId,
            @RequestParam("filter") Option<String> filter,
            @RequestParam("prefix") Option<String> resourceNamePrefix,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewResourcesManager.findResourcesByStaffIds(
                uid, officeId, filter, resourceNamePrefix,
                lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public ResourcesInfo findAvailableResources(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("query") Option<String> query,
            @RequestParam(value = "officeId", required = false) ListF<Long> officeIds,
            @RequestParam(value = "filter", required = false) ListF<String> filters,
            @RequestParam(value = "email", required = false) ListF<Email> emails,
            @BindJson IntervalAndRepetitionData intervalData,
            @AvailParameters.Bind AvailParameters parameters,
            @RequestParam("limit") Option<Integer> limit,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewResourcesManager.findAvailableResources(
                uid, new MultiofficeResourceFilter(officeIds, filters, query, emails),
                intervalData, limit.getOrElse(10), parameters, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public ResourcesInfo findAvailableResourcesByStaffId(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("query") Option<String> query,
            @RequestParam(value = "officeId", required = false) ListF<Long> officeIds,
            @RequestParam(value = "filter", required = false) ListF<String> filters,
            @RequestParam(value = "email", required = false) ListF<Email> emails,
            @BindJson IntervalAndRepetitionData intervalData,
            @AvailParameters.Bind AvailParameters parameters,
            @RequestParam("limit") Option<Integer> limit,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewResourcesManager.findAvailableResourcesByStaffIds(
                uid, officeIds, filters, query, emails,
                intervalData, limit.getOrElse(10), parameters, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public ResourcesSchedule getResourcesSchedule(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("officeId") ListF<Long> officeIds,
            @RequestParam(value = "filter", required = false) ListF<String> filters,
            @RequestParam("prefix") Option<String> prefix,
            @RequestParam(value = "email", required = false) ListF<Email> emails,
            @RequestParam("from") Option<WebDate> from,
            @RequestParam("to") Option<WebDate> till,
            @RequestParam("date") Option<LocalDate> date,
            @RequestParam("bookableOnly") Option<Boolean> bookableOnly,
            @RequestParam("exceptEventId") Option<Long> exceptEventId,
            @RequestParam("tz") Option<DateTimeZone> tz,
            @RequestParam("lang") Option<Language> lang)
    {
        ValidateParam.isTrue("from/to", from.isPresent() == till.isPresent(), "From and to must be both present or not");
        ValidateParam.isTrue("from/to", from.isPresent() != date.isPresent(), "Specify either date or from/to");

        ValidateParam.isFalse("from/to", from.exists(f -> till.exists(t ->
                f.toInstant(DateTimeZone.UTC).isAfter(t.toInstantNextDay(DateTimeZone.UTC)))), "From is after to");
        return webNewResourcesManager.getResourcesSchedule(
                uid, new MultiofficeResourceFilter(officeIds, filters, prefix, emails),
                from.orElse(date.map(WebDate::localDate)).get(), till.orElse(date.map(WebDate::localDate)).get(),
                bookableOnly.getOrElse(false), exceptEventId, tz, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public ResourcesSchedule getResourcesScheduleByStaffId(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("officeId") ListF<Long> officeIds,
            @RequestParam(value = "filter", required = false) ListF<String> filters,
            @RequestParam("prefix") Option<String> prefix,
            @RequestParam(value = "email", required = false) ListF<Email> emails,
            @RequestParam("from") Option<WebDate> from,
            @RequestParam("to") Option<WebDate> till,
            @RequestParam("date") Option<LocalDate> date,
            @RequestParam("bookableOnly") Option<Boolean> bookableOnly,
            @RequestParam("exceptEventId") Option<Long> exceptEventId,
            @RequestParam("tz") Option<DateTimeZone> tz,
            @RequestParam("lang") Option<Language> lang)
    {
        ValidateParam.isTrue("from/to", from.isPresent() == till.isPresent(), "From and to must be both present or not");
        ValidateParam.isTrue("from/to", from.isPresent() != date.isPresent(), "Specify either date or from/to");

        ValidateParam.isFalse("from/to", from.exists(f -> till.exists(t ->
                f.toInstant(DateTimeZone.UTC).isAfter(t.toInstantNextDay(DateTimeZone.UTC)))), "From is after to");
        return webNewResourcesManager.getResourcesScheduleByStaffIds(
                uid, officeIds, filters, prefix, emails,
                from.orElse(date.map(WebDate::localDate)).get(), till.orElse(date.map(WebDate::localDate)).get(),
                bookableOnly.getOrElse(false), exceptEventId, tz, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public UsersAndResourcesInfo findUsersAndResources(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("loginOrEmails") Option<String> loginOrEmailsParameter,
            @RequestParam("lang") Option<Language> lang,
            @BindJson Option<FindUsersAndResourcesData> loginOrEmailsBody)
    {
        Option<String> loginsOrEmails = loginOrEmailsBody
                .flatMapO(FindUsersAndResourcesData::getLoginOrEmails)
                .orElse(loginOrEmailsParameter);

        ValidateParam.isTrue("loginOrEmails", loginsOrEmails.isPresent(), "Parameter loginOrEmails must be provided in body or url");

        return webNewResourcesManager.findUsersAndResources(uid, loginsOrEmails.get(), lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public UserOrResourceInfo getUserOrResourceInfo(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("email") Email email,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewResourcesManager.getUserOrResourceInfo(uid, email, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public WebResourceInfo getResourceDescription(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("email") Email email, @RequestParam("lang") Option<Language> lang)
    {
        return webNewResourcesManager.getResourceDescription(uid, email, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public OfficesInfo getOffices(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("includeOthers") Option<Boolean> includeOthers,
            @RequestParam("tz") Option<DateTimeZone> tz,
            @RequestParam("lang") Option<Language> lang)
    {
        return webNewResourcesManager.getOffices(uid, includeOthers, tz, lang.getOrElse(Language.RUSSIAN));
    }

    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    @Action
    public OfficesTzInfo getOfficesTzOffsets(
            @RequestParam("uid") PassportUid uid,
            @RequestParam("ts") LocalDateTime ts,
            @RequestParam("tz") Option<DateTimeZone> tz)
    {
        return webNewResourcesManager.getOfficesTzOffsets(uid, ts, tz);
    }

}
