package ru.yandex.calendar.frontend.webNew.dto.in;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import javax.annotation.Nullable;

import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.frontend.a3.interceptors.LoggingParameterBinder;
import ru.yandex.commune.a3.action.invoke.ActionInvocationContext;
import ru.yandex.commune.a3.action.parameter.Parameter;
import ru.yandex.commune.a3.action.parameter.ParameterDescriptor;
import ru.yandex.commune.a3.action.parameter.WebRequest;
import ru.yandex.commune.a3.action.parameter.bind.ParameterBinder;
import ru.yandex.commune.a3.action.parameter.bind.annotation.BindWith;
import ru.yandex.commune.a3.utils.ParameterRef;
import ru.yandex.commune.a3.utils.ParameterRetriever;
import ru.yandex.commune.a3.utils.ParameterRetrieverFactory;

/**
 * @author dbrylev
 */
public class AvailParameters {
    public final Option<Instant> instanceStart;
    public final Option<Long> exceptEventId;
    public final Option<DateTimeZone> tz;

    public AvailParameters(Option<Instant> instanceStart, Option<Long> exceptEventId, Option<DateTimeZone> tz) {
        this.instanceStart = instanceStart;
        this.exceptEventId = exceptEventId;
        this.tz = tz;
    }

    public static AvailParameters empty() {
        return new AvailParameters(Option.empty(), Option.empty(), Option.empty());
    }

    public static AvailParameters eventId(long eventId) {
        return new AvailParameters(Option.empty(), Option.of(eventId), Option.empty());
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target({ElementType.FIELD, ElementType.PARAMETER})
    @BindWith(value = Binder.class)
    public @interface Bind {
    }

    public static class Binder implements ParameterBinder, LoggingParameterBinder {
        private static final ParameterRetriever<LocalDateTime> instanceStart = ParameterRetrieverFactory.cons(
                LocalDateTime.class, ParameterRef.request("instanceStartTs"));

        private static final ParameterRetriever<Long> exceptEventId = ParameterRetrieverFactory.cons(
                Long.class, ParameterRef.request("exceptEventId"));

        private static final ParameterRetriever<String> tz = ParameterRetrieverFactory.cons(
                String.class, ParameterRef.request("tz"));

        @Nullable
        @Override
        public Object createAndBind(WebRequest request, ActionInvocationContext context, ParameterDescriptor descriptor) {
            return new AvailParameters(
                    instanceStart.retrieve(request, context).map(dt -> dt.toDateTime(DateTimeZone.UTC).toInstant()),
                    exceptEventId.retrieve(request, context),
                    tz.retrieve(request, context).map(DateTimeZone::forID));
        }

        @Override
        public Option<String> getNameForLogging(Parameter parameter) {
            return parameter.getValue().map(p -> "parameters");
        }

        @Override
        public Option<String> getValueForLogging(Parameter parameter) {
            return parameter.getValue().filterByType(AvailParameters.class).map(p -> {
                Tuple2List<String, String> fields = Tuple2List.arrayList();

                p.instanceStart.forEach(i -> fields.add("instanceStartTs", i.toString()));
                p.exceptEventId.forEach(i -> fields.add("exceptEventId", Long.toString(i)));
                p.tz.forEach(i -> fields.add("tz", i.getID()));

                return "[" + fields.mkString(", ", "=") + "]";
            });
        }
    }
}
