package ru.yandex.calendar.frontend.webNew.dto.in;

import java.util.Map;
import java.util.Optional;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.Setter;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Layer;
import ru.yandex.calendar.logic.beans.generated.LayerUser;
import ru.yandex.calendar.logic.layer.LayerRoutines;
import ru.yandex.calendar.logic.layer.LayerType;
import ru.yandex.calendar.logic.layer.LayerUserCssClassHack;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.sharing.perm.LayerActionClass;
import ru.yandex.calendar.util.color.Color;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.email.Email;

import static java.util.Collections.emptyMap;

/**
 * @author dbrylev
 */
@AllArgsConstructor
@Setter
@Getter
@Builder(toBuilder = true)
@BenderBindAllFields
public class LayerData {
    private Option<String> color;
    private Option<Integer> defaultEventsDurationMinutes;
    private ListF<Notification> notifications;
    private Option<Boolean> notifyAboutEventChanges;

    private Option<Boolean> affectAvailability;

    private Option<LayerType> type;

    private Option<String> feedUrl;

    /* owner related */
    private Option<String> name;
    private Option<Boolean> isDefault;

    @Deprecated
    private Optional<Boolean> isClosed;  //unused, just for backwards compatibility
    private Optional<Boolean> isEventsClosedByDefault;
    @BenderPart(wrapperName = "participants")
    private Option<ListF<Participant>> participants;

    public static LayerData empty() {
        return new LayerData(
                Option.empty(), Option.empty(), Cf.list(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Optional.empty(), Optional.empty(), Option.empty());
    }

    @AllArgsConstructor
    @Getter
    @BenderBindAllFields
    public static class Participant {
        private Email email;
        private LayerActionClass permission;
    }

    public Option<String> getName() {
        return name;
    }

    public LayerUser getLayerUser() {
        LayerUser data = new LayerUser();
        color.map(Color::parseRgb).forEach(clr -> {
            data.setHeadBgColor(clr.getRgb());
            data.setCssClass(LayerUserCssClassHack.getBestMatchingCssClassByColor(clr));
        });
        data.setAffectsAvailability(affectAvailability.orElse(true));
        data.setIsNotifyChanges(notifyAboutEventChanges.orElse(false));
        data.setDefaultEventsDurationMinutes(defaultEventsDurationMinutes.orElse(30));

        return data;
    }

    public ListF<Notification> getNotifications() {
        return notifications;
    }

    public Layer getLayer(LayerType type, boolean forCreate) {
        Layer data = new Layer();
        isEventsClosedByDefault.ifPresent(data::setIsEventsClosedByDefault);

        if (!name.isPresent()) {
            if (forCreate) {
                ValidateParam.some("layer", name, "name is not set");
            }
        } else if (type == LayerType.USER) {
            data.setName(name.filterNot(LayerRoutines.DEFAULT_USER_LAYER_NAME.getAllNames()::containsTs));
        } else {
            data.setName(name.get());
        }

        return data;
    }

    public boolean isDefault() {
        ValidateParam.some("layer", isDefault, "isDefault is not set");

        return isDefault.get();
    }

    public Map<Email, LayerActionClass> getPermissionMapSafe() {
        return participants.isPresent() ? getPermissionMap() : emptyMap();
    }

    public Map<Email, LayerActionClass> getPermissionMap() {
        ValidateParam.some("layer", participants, "participants is not set");

        ListF<Participant> ps = participants.get();

        return StreamEx.of(ps).toMap(Participant::getEmail, Participant::getPermission);
    }

    public Option<Boolean> getIsDefault() {
        return isDefault;
    }

    public Option<LayerType> getTypeO() {
        return type;
    }

    public Option<String> getFeedUrlO() {
        return feedUrl;
    }
}
