package ru.yandex.calendar.frontend.webNew.dto.in;

import lombok.AllArgsConstructor;
import lombok.Setter;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.calendar.frontend.webNew.dto.inOut.RepetitionData;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.resource.ResourceFilter;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author gutman
 * @author Sergey Shinderuk
 */
@AllArgsConstructor
@Setter
@Bendable
public class SuggestData {

    @BenderPart
    private ListF<Email> users;
    @BenderPart(name = "office", wrapperName = "offices")
    private ListF<Office> offices;
    @BenderPart
    private Option<LocalDateTime> searchStart;
    @BenderPart
    private Option<Boolean> searchBackward;
    @BenderPart
    private LocalDateTime eventStart;
    @BenderPart
    private LocalDateTime eventEnd;
    @BenderPart
    private Option<LocalDate> date;
    @BenderPart
    private Option<Direction> direction;
    @BenderPart
    private Option<LocalDateTime> selectedStart;
    @BenderPart
    private Option<RepetitionData> repetition;
    @BenderPart
    private Option<Integer> numberOfOptions;
    @BenderPart
    private Option<Boolean> ignoreUsersEvents;
    @BenderPart
    private Option<Mode> mode;

    public static SuggestData interval(LocalDateTime start, LocalDateTime end) {
        return new SuggestData(
                Cf.list(), Cf.list(), Option.empty(), Option.empty(),
                start, end, Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), Option.empty());
    }

    public ListF<Email> getUsers() {
        return users;
    }

    public ListF<Office> getOffices() {
        return offices;
    }

    public ListF<Long> getOfficeIds() {
        return offices.map(Office.getIdF());
    }

    public MapF<Long, ResourceFilter> getResourceFilters() {
        return offices.toMap(Office.getIdF(), Office.getResourceFilterF());
    }

    public MapF<Long, Integer> getNumbersOfResources() {
        return offices.toMap(Office.getIdF(), Office.getNumberOfResourcesF());
    }

    public Option<LocalDateTime> getSearchStart() {
        return searchStart;
    }

    public Direction getDirection() {
        return direction.getOrElse(Direction.BOTH);
    }

    public Option<Boolean> getSearchBackward() {
        return searchBackward;
    }

    public LocalDateTime getEventStart() {
        return eventStart;
    }

    public LocalDateTime getEventEnd() {
        return eventEnd;
    }

    public Option<LocalDate> getDate() {
        return date;
    }

    public Option<Instant> getSelectedStart(DateTimeZone tz) {
        return selectedStart.map(s -> AuxDateTime.toInstantIgnoreGap(s, tz));
    }

    public Option<Integer> getNumberOfOptions() {
        return numberOfOptions;
    }

    public boolean isIgnoreUsersEvents() {
        return ignoreUsersEvents.getOrElse(false);
    }

    public Duration getDuration() {
        return new Period(eventStart, eventEnd).toStandardDuration();
    }

    public Option<RepetitionData> getRepetitionData() {
        return repetition;
    }

    public InstantInterval getEventInterval(DateTimeZone tz) {
        return new InstantInterval(getEventStart(tz), getDuration());
    }

    public Instant getEventStart(DateTimeZone tz) {
        return AuxDateTime.toInstantIgnoreGap(eventStart, tz);
    }

    public Instant getSelectedOrEventStart(DateTimeZone tz) {
        return getSelectedStart(tz).getOrElse(getEventStart(tz));
    }

    public Option<Repetition> getRepetition(DateTimeZone tz) {
        return repetition.isPresent()
                ? Option.of(repetition.get().toRepetition(getEventStart(tz), tz))
                : Option.<Repetition>empty();
    }

    public Mode getMode() {
        return mode.getOrElse(Mode.ANY_ROOM);
    }

    public boolean isSameRoomMode() {
        return getMode() == Mode.SAME_ROOM;
    }

    @AllArgsConstructor
    @Bendable
    public static class Office {

        @BenderPart
        private long id;
        @BenderPart
        private ListF<Email> selectedResourceEmails;
        @BenderPart
        private Option<Integer> numberOfResources;
        @BenderPart
        private Option<String> filter;

        public long getId() {
            return id;
        }

        public ListF<Email> getSelectedResourceEmails() {
            return selectedResourceEmails;
        }

        public int getNumberOfResources() {
            return numberOfResources.getOrElse(1);
        }

        public ResourceFilter getResourceFilter() {
            if (filter.isPresent() && !filter.get().isEmpty()) {
                return ResourceFilter.any().withFilter(filter.get());
            } else {
                return ResourceFilter.any();
            }
        }

        public static Function<Office, Long> getIdF() {
            return new Function<Office, Long>() {
                public Long apply(Office office) {
                    return office.getId();
                }
            };
        }

        public static Function<Office, ListF<Email>> getSelectedResourceEmailsF() {
            return new Function<Office, ListF<Email>>() {
                public ListF<Email> apply(Office office) {
                    return office.getSelectedResourceEmails();
                }
            };
        }

        public static Function<Office, ResourceFilter> getResourceFilterF() {
            return new Function<Office, ResourceFilter>() {
                public ResourceFilter apply(Office office) {
                    return office.getResourceFilter();
                }
            };
        }

        public static Function<Office, Integer> getNumberOfResourcesF() {
            return new Function<Office, Integer>() {
                public Integer apply(Office office) {
                    return office.getNumberOfResources();
                }
            };
        }
    }

    public static enum Mode {
        ANY_ROOM,
        SAME_ROOM,
        ;
    }

    public enum Direction {
        FORWARD,
        BACKWARD,
        BOTH,
        ;

        public boolean containsForward() {
            return this != BACKWARD;
        }

        public boolean containsBackward() {
            return this != FORWARD;
        }
    }
}
