package ru.yandex.calendar.frontend.webNew.dto.in;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.Setter;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.webNew.WebNewCodecs;
import ru.yandex.calendar.frontend.webNew.dto.inOut.NoNotificationsRange;
import ru.yandex.calendar.frontend.webNew.dto.inOut.TodoEmailTimes;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.event.grid.ViewType;
import ru.yandex.calendar.logic.sharing.InvAcceptingType;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.misc.bender.Format;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BendingMethod;
import ru.yandex.misc.bender.internal.pojo.PojoUnmarshaller;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.FieldsSettersBasedPojoUnmarshallerFactory;
import ru.yandex.misc.bender.parse.MainUnmarshallerFactory;
import ru.yandex.misc.bender.parse.UnmarshallerContext;
import ru.yandex.misc.bender.parse.XmlOrJsonNode;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.reflection.ClassX;

/**
 * @author dbrylev
 */
@AllArgsConstructor
@Getter
@Setter
@BenderBindAllFields
public class UserSettingsData {
    private Option<String> tz;
    private Option<String> lastOfferedGeoTz;

    private Option<Boolean> transliterateSms;

    private Option<NoNotificationsRange> noNotificationsRange;
    private Option<TodoEmailTimes> todoEmailTimes;

    private Option<String> weekStartDay;
    private Option<Integer> dayStartHour;
    private Option<String> defaultView;

    private Option<Boolean> autoAcceptEventInvitations;
    private Option<Boolean> showTodosInGrid;
    private Option<Boolean> showAvailabilityToAnyone;

    private Option<Boolean> useYaMaps;
    private Option<Boolean> showShortForm;

    private Option<Boolean> remindUndecided;
    private Option<Boolean> letParticipantsEdit;
    private Option<Boolean> noNotificationsInYaMail;
    private Option<Boolean> noNotificationsDuringAbsence;

    private Option<Email> email;

    public Option<String> getTz() {
        return tz;
    }

    public Option<String> getLastOfferedGeoTz() {
        return lastOfferedGeoTz;
    }

    public Settings getSettings(DateTimeZone userTz) {
        Settings settings = new Settings();

        transliterateSms.forEach(settings::setTranslitSms);

        noNotificationsRange.forEach(s -> {
            DateTimeZone dataTz = tz.map(DateTimeZone::forID).getOrElse(userTz);

            settings.setNoNtfStartTs(s.getSinceDate().map(AuxDateTime.dayStartF(dataTz)));
            settings.setNoNtfEndTs(s.getUntilDate().map(AuxDateTime.dayStartF(dataTz)));

            settings.setNoNtfStartTm(s.getFromTime().map(AuxDateTime.localTimeMillisOfDayF()));
            settings.setNoNtfEndTm(s.getToTime().map(AuxDateTime.localTimeMillisOfDayF()));
        });

        todoEmailTimes.forEach(s -> {
            settings.setTodoPlannedEmailTm(s.getPlanned().map(AuxDateTime.localTimeMillisOfDayF()));
            settings.setTodoExpiredEmailTm(s.getExpired().map(AuxDateTime.localTimeMillisOfDayF()));
        });

        weekStartDay.forEach(s -> settings.setStartWeekday(DayOfWeek.R.fromValue(s)));
        dayStartHour.forEach(settings::setGridTopHours);
        defaultView.forEach(s -> settings.setViewType(ViewType.R.fromValue(s)));

        autoAcceptEventInvitations.forEach(s -> settings.setInvAcceptType(
                s ? InvAcceptingType.AUTO : InvAcceptingType.MANUAL));

        showTodosInGrid.forEach(settings::setShowTodo);
        showAvailabilityToAnyone.forEach(settings::setShowAvailability);
        useYaMaps.forEach(settings::setMapsEnabled);
        showShortForm.forEach(settings::setShowShortForm);

        email.forEach(settings::setEmail);

        return settings;
    }

    public SettingsYt getSettingsYt() {
        SettingsYt settings = new SettingsYt();

        remindUndecided.forEach(settings::setRemindUndecided);
        letParticipantsEdit.forEach(settings::setLetParticipantsEdit);
        noNotificationsInYaMail.forEach(s -> settings.setXivaReminderEnabled(!s));
        noNotificationsDuringAbsence.forEach(settings::setNoNtfDuringAbsence);

        return settings;
    }

    private static final PojoUnmarshaller unmarshaller = new FieldsSettersBasedPojoUnmarshallerFactory()
            .createUnmarshaller(
                    WebNewCodecs.getConfiguration().getSettings(), Format.JSON,
                    new MainUnmarshallerFactory(Format.JSON, WebNewCodecs.getConfiguration()),
                    ClassX.wrap(UserSettingsData.class).asType());

    @BendingMethod
    public static UserSettingsData parse(XmlOrJsonNode node, UnmarshallerContext context) {
        UserSettingsData data = unmarshaller.parseJsonNode(node.getJson(), context)
                .<UserSettingsData>cast().getOrThrow();

        Function1B<String> hasField = field -> node.getJson().getField(field).exists(BenderJsonNode::isObject);

        if (!data.noNotificationsRange.isPresent() && hasField.apply("noNotificationsRange")) {
            data.noNotificationsRange = Option.of(new NoNotificationsRange(
                    Option.empty(), Option.empty(), Option.empty(), Option.empty()));
        }
        if (!data.todoEmailTimes.isPresent() && hasField.apply("todoEmailTimes")) {
            data.todoEmailTimes = Option.of(new TodoEmailTimes(Option.empty(), Option.empty()));
        }
        return data;
    }
}
