package ru.yandex.calendar.frontend.webNew.dto.in;

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.Setter;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.bender.RawJson;
import ru.yandex.calendar.frontend.bender.WebDateTime;
import ru.yandex.calendar.frontend.webNew.dto.inOut.EventAttachmentData;
import ru.yandex.calendar.frontend.webNew.dto.inOut.RepetitionData;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.EventFields;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.event.model.EventUserData;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.sharing.perm.EventActionClass;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.inside.passport.PassportSid;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.email.Email;

@AllArgsConstructor
@Setter
@Getter
@Bendable
public class WebEventData {
    @BenderPart
    private Option<EventType> type;
    @BenderPart
    private Option<Integer> sid;
    @BenderPart
    private Option<Boolean> isAllDay;
    @BenderPart
    private Option<WebDateTime> startTs;
    @BenderPart
    private Option<WebDateTime> endTs;
    @BenderPart
    private Option<String> name;
    @BenderPart
    private Option<String> location;
    @BenderPart
    private Option<String> description;
    @BenderPart(name = "attachment", wrapperName = "attachments")
    private Option<ListF<EventAttachmentData>> attachments;
    @BenderPart
    private Option<String> mapUrl;
    @BenderPart
    private Option<String> url;


    @BenderPart
    private Option<Boolean> participantsCanInvite;
    @BenderPart
    private Option<Boolean> participantsCanEdit;
    @BenderPart
    private Option<Boolean> othersCanView;

    @BenderPart
    private Option<Email> organizer;

    @BenderPart(name = "attendee", wrapperName = "attendees")
    private Option<ListF<Email>> attendeeEmails;
    @BenderPart(name = "optionalAttendee", wrapperName = "optionalAttendees")
    private Option<ListF<Email>> optionalAttendeeEmails;

    @BenderPart
    private Option<RepetitionData> repetition;

    @BenderFlatten
    private WebEventUserData userData;

    @BenderPart
    private Option<RawJson> eventData;
    @BenderPart
    private Option<String> conferenceUrl;

    public WebEventData(Event event, DateTimeZone eventTz, Option<Email> organizer,
                        Option<ListF<Email>> attendees, Option<ListF<Email>> optionalAttendees,
                        Option<RepetitionData> repetition, WebEventUserData userData,
                        Option<ListF<EventAttachmentData>> attachments) {
        this(event.getFieldValueO(EventFields.TYPE),
            event.getFieldValueO(EventFields.SID).map(PassportSid::getValue),
            event.getFieldValueO(EventFields.IS_ALL_DAY),
            event.getFieldValueO(EventFields.START_TS).map(v -> WebDateTime.dateTime(v.toDateTime(eventTz))),
            event.getFieldValueO(EventFields.END_TS).map(v -> WebDateTime.dateTime(v.toDateTime(eventTz))),
            event.getFieldValueO(EventFields.NAME),
            event.getFieldValueO(EventFields.LOCATION),
            event.getFieldValueO(EventFields.DESCRIPTION),
            attachments,
            Option.empty(),
            event.getUrl(),
            event.getFieldValueO(EventFields.PARTICIPANTS_INVITE),
            event.getFieldValueO(EventFields.PERM_PARTICIPANTS).map(perm -> perm == EventActionClass.EDIT),
            event.getFieldValueO(EventFields.PERM_ALL).map(v -> v == EventActionClass.VIEW || v == EventActionClass.EDIT),
            organizer,
            attendees,
            optionalAttendees,
            repetition,
            userData,
            event.getData(),
            event.getFieldValueO(EventFields.CONFERENCE_URL));
    }

    public static WebEventData empty() {
        return new WebEventData(
                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), Option.empty(), Option.empty(), Option.empty(), Option.empty(),
                Option.empty(), WebEventUserData.empty(), Option.empty(), Option.empty());
    }

    public Event getEvent(DateTimeZone timeZone, DateTimeZone eventTz) {
        Event e = new Event();
        type.forEach(e::setType);
        e.setSid(sid.map(PassportSid::cons).getOrElse(PassportSid.CALENDAR));

        e.setIsAllDay(isAllDay.getOrElse(Boolean.FALSE));

        startTs.forEach(i -> e.setStartTs(getStartEndTs(i, timeZone, eventTz)));
        endTs.forEach(i -> e.setEndTs(getStartEndTs(i, timeZone, eventTz)));

        name.forEach(e::setName);
        description.forEach(e::setDescription);

        location.forEach(e::setLocation);
        url.forEach(e::setUrl);
        conferenceUrl.forEach(e::setConferenceUrl);

        participantsCanInvite.forEach(e::setParticipantsInvite);
        participantsCanEdit.forEach(p -> e.setPermParticipants(p ? EventActionClass.EDIT : EventActionClass.VIEW));
        othersCanView.forEach(p -> e.setPermAll(p ? EventActionClass.VIEW : EventActionClass.NONE));
        e.setData(eventData);

        return e;
    }

    private Instant getStartEndTs(WebDateTime dateTime, DateTimeZone dataTz, DateTimeZone eventTz) {
        return dateTime.toInstant(isAllDay.isSome(true) ? eventTz : dataTz);
    }

    public Option<ListF<Notification>> getNotifications() {
        return userData.getNotifications();
    }

    public EventUserData getEventUserData() {
        return userData.getEventUserData();
    }

    public Option<Email> getOrganizerEmail() {
        return organizer;
    }

    public Option<Repetition> getRepetition(DateTimeZone dataTz, DateTimeZone eventTz) {
        if (repetition.isPresent()) {
            ValidateParam.some("startTs", startTs, "Need to know startTs to calculate repetition");
            return Option.of(repetition.get().toRepetition(getStartEndTs(startTs.get(), dataTz, eventTz), eventTz));
        } else {
            return Option.empty();
        }
    }

    public Option<PassportSid> getSid() {
        return sid.map(PassportSid::cons);
    }

    public Option<Boolean> getAllDay() {
        return isAllDay;
    }

    public Option<Long> getLayerId() {
        return userData.getLayerId();
    }

    public Option<ListF<EventAttachmentData>> getAttachments() {
        return attachments;
    }

    public void setRepetitionNone() {
        this.repetition = Option.empty();
    }

    public void setEventData(Option<String> json) {
        this.eventData = json.map(RawJson::new);
    }
}
