package ru.yandex.calendar.frontend.webNew.dto.inOut;

import lombok.EqualsAndHashCode;
import lombok.Setter;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.repetition.RegularRepetitionRule;
import ru.yandex.calendar.util.dates.DayOfWeek;
import ru.yandex.commune.a3.action.parameter.ValidateParam;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.Validate;

@Setter
@EqualsAndHashCode
@BenderBindAllFields
public class RepetitionData {
    private RegularRepetitionRule type;
    private int each;
    private Option<String> weeklyDays;
    private Option<Boolean> monthlyLastweek;
    private Option<LocalDate> dueDate;

    public Repetition toRepetition(Instant eventStartTs, DateTimeZone tz) {
        Repetition repetition = new Repetition();
        repetition.setType(type);
        repetition.setREach(each);

        ValidateParam.equals("repetition", weeklyDays.isPresent(), type == RegularRepetitionRule.WEEKLY);
        ValidateParam.isTrue("repetition", !monthlyLastweek.isPresent() || type == RegularRepetitionRule.MONTHLY_DAY_WEEKNO);

        if (type == RegularRepetitionRule.WEEKLY) {
            if (!weeklyDays.get().isEmpty()) {
                ListF<DayOfWeek> ds = Cf.x(weeklyDays.get().split(",")).map(DayOfWeek.byNameF());

                repetition.setRWeeklyDays(ds.unique().sorted().map(DayOfWeek.getDbValueF()).mkString(","));
            } else {
                repetition.setType(RegularRepetitionRule.NONE);
            }
        }
        if (type == RegularRepetitionRule.MONTHLY_DAY_WEEKNO) {
            repetition.setRMonthlyLastweek(monthlyLastweek.getOrElse(false));
        }

        if (dueDate.isPresent()) {
            repetition.setDueTs(EventRoutines.calculateDueTsFromUntilDate(eventStartTs, dueDate.get(), tz));
        } else {
            repetition.setDueTs(Option.empty());
        }

        return repetition;
    }

    public static RepetitionData fromRepetition(Repetition repetition, DateTimeZone tz) {
        Validate.isTrue(repetition.getType() != RegularRepetitionRule.NONE);

        RepetitionData data = new RepetitionData();
        data.type = repetition.getType();
        data.each = repetition.getREach().get();
        if (repetition.getDueTs().isPresent()) {
            LocalDate dueDate = EventRoutines.convertDueTsToUntilDate(repetition.getDueTs().get(), tz);
            data.dueDate = Option.of(dueDate);
        } else {
            data.dueDate = Option.empty();
        }
        data.weeklyDays = repetition.getRWeeklyDays();
        data.monthlyLastweek = repetition.getRMonthlyLastweek();

        return data;
    }
}
