package ru.yandex.calendar.frontend.webNew.dto.out;

import lombok.Getter;
import lombok.Value;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.bender.WebDateTime;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.util.dates.DateInterval;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.email.Email;

@Value
@BenderBindAllFields
public class ResourcesSchedule {
    ListF<Office> offices;

    public ListF<Resource> getResources() {
        return offices.flatMap(Office::getResources);
    }

    public ListF<Email> getEmails() {
        return getResources().map(r -> r.info.getEmail());
    }

    public Option<Resource> find(Email email) {
        return getResources().find(r -> r.info.getEmail().equalsIgnoreCase(email));
    }

    @Value
    @BenderBindAllFields
    public static class Office {
        long id;
        Option<Long> staffId;
        String name;
        String tzId;
        ListF<Resource> resources;
    }

    @Value
    @BenderBindAllFields
    public static class Resource {
        ResourceInfo info;
        ListF<RestrictionInterval> restrictions;
        ListF<ResourceEvent> events;

        public ListF<LocalDateTime> getEventStarts() {
            return events.map(r -> r.getStart().toLocalDateTime());
        }
    }

    @Value
    @BenderBindAllFields
    public static class ResourceInfo {
        long id;
        String name;
        String alterName;
        Email email;
        ResourceType type;

        boolean hasPhone;
        boolean hasVideo;
        Option<Integer> floor;
        Option<String> group;

        Option<Boolean> canAdmin;
        Option<Boolean> canBook;
        Option<String> protection;
        Option<String> mapUrl;
    }

    @Getter
    @BenderBindAllFields
    public static class ResourceEvent {
        private final Option<Long> eventId;
        private final Option<Long> reservationId;
        private final Option<String> author;
        private final Option<WebUserInfo> authorInfo;

        private final WebDateTime start;
        private final WebDateTime end;

        public ResourceEvent(
                Option<Long> eventId, Option<Long> reservationId, Option<WebUserInfo> authorInfo,
                WebDateTime start, WebDateTime end)
        {
            this.eventId = eventId;
            this.reservationId = reservationId;
            this.authorInfo = authorInfo;
            this.author = authorInfo.filterMap(WebUserInfo::getLogin);
            this.start = start;
            this.end = end;
        }

        public LocalDateTimeInterval getInterval() {
            return new LocalDateTimeInterval(start.toLocalDateTime(), end.toLocalDateTime());
        }
    }

    @Value
    @BenderBindAllFields
    public static class RestrictionInterval {
        Option<WebDateTime> start;
        Option<WebDateTime> end;
        Option<String> title;

        public static RestrictionInterval cons(DateInterval interval, DateTimeZone fromTz, DateTimeZone tz, Language lang) {
            return new RestrictionInterval(
                    interval.start.map(st -> WebDateTime.dateTime(st.toDateTime(fromTz).toDateTime(tz))),
                    interval.end.map(en -> WebDateTime.dateTime(en.toDateTime(fromTz).toDateTime(tz))),
                    interval.title.map(t -> t.getName(lang)));
        }
    }
}
