package ru.yandex.calendar.frontend.webNew.dto.out;

import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.resource.ResourceInaccessibility;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.email.Email;

/**
* @author gutman
*/
@Bendable
public class SubjectAvailability {
    @BenderPart
    private final Email email;
    @BenderFlatten
    private AvailabilityInfo availability;
    @BenderIgnore
    private final Option<ResourceInaccessibility> inaccessibility;

    private SubjectAvailability(Email email, String availability) {
        this(email, availability, Option.empty(), Option.empty(), Option.empty());
    }

    private SubjectAvailability(
            Email email, String availability,
            Option<LocalDate> dueDate, Option<Integer> availableRepetitions,
            Option<ResourceInaccessibility> inaccessibility)
    {
        this.email = email;
        this.availability = new AvailabilityInfo(Option.of(availability), dueDate, availableRepetitions);
        this.inaccessibility = inaccessibility;
    }

    public Email getEmail() {
        return email;
    }

    public String getAvailability() {
        return availability.getAvailability().get();
    }

    public Option<LocalDate> getDueDate() {
        return availability.getDueDate();
    }

    public AvailabilityInfo getAvailabilityInfo() {
        return availability;
    }

    public boolean isAvailable() {
        return availability.getAvailability().isSome(Availability.AVAILABLE.value());
    }

    public boolean isBusy() {
        return availability.getAvailability().isSome(Availability.BUSY.value());
    }

    public static SubjectAvailability availableDue(Email email, LocalDate dueDate, int availableRepetitions) {
        return new SubjectAvailability(
                email, Availability.AVAILABLE.value(),
                Option.of(dueDate), Option.of(availableRepetitions), Option.empty());
    }

    public static SubjectAvailability inaccessible(ResourceInaccessibility inaccessibility) {
        return new SubjectAvailability(
                inaccessibility.getEmail(), Availability.BUSY.value(),
                Option.empty(), Option.empty(), Option.of(inaccessibility));
    }

    public static SubjectAvailability known(Email email, Availability availability) {
        return new SubjectAvailability(email, availability.value());
    }

    public static SubjectAvailability unknown(Email email) {
        return new SubjectAvailability(email, "unknown");
    }

    public Option<ResourceInaccessibility> getInaccessibility() {
        return inaccessibility;
    }


    public static Function2<Email, Availability, SubjectAvailability> knownF() {
        return new Function2<Email, Availability, SubjectAvailability>() {
            public SubjectAvailability apply(Email email, Availability availability) {
                return known(email, availability);
            }
        };
    }

    public static Function<Email, SubjectAvailability> unknownF() {
        return new Function<Email, SubjectAvailability>() {
            public SubjectAvailability apply(Email email) {
                return unknown(email);
            }
        };
    }

    public static Function<Email, SubjectAvailability> notActiveF() {
        return new Function<Email, SubjectAvailability>() {
            public SubjectAvailability apply(Email email) {
                return new SubjectAvailability(email, "not-active");
            }
        };
    }

    public static Function<SubjectAvailability, Email> getEmailF() {
        return new Function<SubjectAvailability, Email>() {
            public Email apply(SubjectAvailability a) {
                return a.email;
            }
        };
    }

    public static Function1B<SubjectAvailability> isAvailableF() {
        return SubjectAvailability::isAvailable;
    }

    public static Function1B<SubjectAvailability> isBusyF() {
        return SubjectAvailability::isBusy;
    }

}
