package ru.yandex.calendar.frontend.webNew.dto.out;

import lombok.Getter;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.calendar.frontend.bender.WebDateTime;
import ru.yandex.calendar.frontend.webNew.dto.in.AvailShapeType;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.event.EventActions;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.avail.AvailabilityIntervals;
import ru.yandex.calendar.logic.event.avail.AvailabilityQueryRefusalReason;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.sharing.participant.Participants;
import ru.yandex.calendar.logic.sharing.participant.YandexUserParticipantInfo;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.inside.passport.AbstractPassportUid;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.enums.EnumUtils;

/**
* @author gutman
*/
@Getter
@Bendable
public class SubjectAvailabilityIntervals {
    @BenderPart
    private final Email email;
    @BenderPart
    private final String status;
    @BenderPart(name = "interval", wrapperName = "intervals")
    private final Option<ListF<AvailabilityInterval>> intervals;

    private SubjectAvailabilityIntervals(
            Email email, Either<ListF<AvailabilityInterval>, AvailabilityQueryRefusalReason> result)
    {
        this.email = email;
        this.status = result.isLeft() ? "ok" : EnumUtils.toXmlName(result.getRight());
        this.intervals = result.leftO();
    }

    public SubjectAvailabilityIntervals(Email email, ListF<AvailabilityInterval> intervals) {
        this(email, Either.<ListF<AvailabilityInterval>, AvailabilityQueryRefusalReason>left(intervals));
    }

    public SubjectAvailabilityIntervals(Email email, AvailabilityQueryRefusalReason refusalReason) {
        this(email, Either.<ListF<AvailabilityInterval>, AvailabilityQueryRefusalReason>right(refusalReason));
    }

    public static Function2<Email, AvailabilityIntervals, SubjectAvailabilityIntervals> intervalsF(
            AvailShapeType shape, DateTimeZone tz, Language lang)
    {
        return (email, intervals) -> new SubjectAvailabilityIntervals(
                email, intervals.unmerged().map(AvailabilityInterval.convertF(shape, tz, lang)));
    }

    public static Function2<Email, AvailabilityQueryRefusalReason, SubjectAvailabilityIntervals> refusalF() {
        return new Function2<Email, AvailabilityQueryRefusalReason, SubjectAvailabilityIntervals>() {
            public SubjectAvailabilityIntervals apply(Email email, AvailabilityQueryRefusalReason refusalReason) {
                return new SubjectAvailabilityIntervals(email, refusalReason);
            }
        };
    }

    @Getter
    @Bendable
    public static class AvailabilityInterval {
        @BenderPart
        private final Availability availability;
        @BenderPart
        private Option<EventType> eventType;
        @BenderPart
        private final WebDateTime start;
        @BenderPart
        private final WebDateTime end;
        @BenderPart
        private final Option<Boolean> isAllDay;
        @BenderPart
        private final Option<LocalDateTime> instanceStart;
        @BenderPart
        private final Option<Long> eventId;
        @BenderPart
        private final Option<String> eventName;
        @BenderPart
        private final Option<WebUserParticipantInfo> organizer;
        @BenderPart(wrapperName = "attendees")
        private final Option<ListF<WebUserParticipantInfo>> attendees;
        @BenderPart(wrapperName = "resources")
        private final Option<ListF<AvailabilityIntervalResourceInfo>> resources;
        @BenderPart(wrapperName = "actions")
        private final Option<EventActions> actions;
        @BenderPart(wrapperName = "sequence")
        private final Option<Integer> sequence;

        public AvailabilityInterval(
                Availability availability, Option<EventType> eventType,
                WebDateTime start, WebDateTime end, Option<Boolean> isAllDay, Instant instanceStart,
                Option<Long> eventId, Option<String> eventName,
                Option<WebUserParticipantInfo> organizer,
                Option<ListF<WebUserParticipantInfo>> attendees,
                Option<ListF<AvailabilityIntervalResourceInfo>> resources,
                Option<EventActions> eventActions, Option<Integer> sequence)
        {
            this.availability = availability;
            this.eventType = eventType;
            this.start = start;
            this.end = end;
            this.isAllDay = isAllDay;
            this.instanceStart = Option.when(eventName.isPresent(), new LocalDateTime(instanceStart, DateTimeZone.UTC));
            this.eventId = eventId;
            this.eventName = eventName;
            this.organizer = organizer;
            this.attendees = attendees;
            this.resources = resources;
            this.actions = eventActions;
            this.sequence = sequence;
        }

        public static Function<ru.yandex.calendar.logic.event.avail.AvailabilityInterval, AvailabilityInterval>
                convertF(AvailShapeType shape, DateTimeZone tz, Language lang)
        {
            return new Function<ru.yandex.calendar.logic.event.avail.AvailabilityInterval, AvailabilityInterval>() {
                public AvailabilityInterval apply(ru.yandex.calendar.logic.event.avail.AvailabilityInterval i) {
                    ListF<ParticipantInfo> attendees = i.getParticipants()
                            .flatMap(Participants::getAttendeesButNotOrganizerSafe);

                    return new AvailabilityInterval(
                            i.getAvailability(), i.getEventType(),
                            WebDateTime.dateTime(i.getInterval().getStart().toDateTime(tz)),
                            WebDateTime.dateTime(i.getInterval().getEnd().toDateTime(tz)),
                            i.getIsAllDay(),
                            i.getInterval().getStart(),
                            i.getEventId(),
                            i.getEventName(),
                            i.getOrganizer().map(s -> new WebUserParticipantInfo(userInfo(s), Decision.YES)),
                            Option.when(shape.isWithAttendees(), attendees.filterMap(userAttendeeF())),
                            Option.when(shape.isWithResources(), i.getResources()
                                    .map(AvailabilityIntervalResourceInfo.consF(lang))),
                            i.getActions(),
                            i.getSequenceId());
                }
            };
        }

        public static WebUserInfo userInfo(SettingsInfo settings) {
            return new WebUserInfo(
                    Option.of(settings.getUid().getUid()),
                    settings.getUserNameIfYt().getOrElse(""),
                    settings.getEmail(), settings.getUserLogin(),
                    settings.getYt().filterMap(SettingsYt.getActiveOfficeIdF()));
        }

        public static Option<WebUserParticipantInfo> userAttendee(ParticipantInfo p) {
            if (!p.getId().isAnyUser()) return Option.empty();

            WebUserInfo info = p instanceof YandexUserParticipantInfo
                    ? userInfo(((YandexUserParticipantInfo) p).getSettings())
                    : new WebUserInfo(p.getUid().map(AbstractPassportUid::getUid), p.getName(), p.getEmail(), Option.empty(), Option.empty());

            return Option.of(new WebUserParticipantInfo(info, p.getDecision()));
        }

        public static Function<ParticipantInfo, Option<WebUserParticipantInfo>> userAttendeeF() {
            return new Function<ParticipantInfo, Option<WebUserParticipantInfo>>() {
                public Option<WebUserParticipantInfo> apply(ParticipantInfo p) {
                    return userAttendee(p);
                }
            };
        }
    }

    @BenderBindAllFields
    public static class AvailabilityIntervalResourceInfo {
        public final String name;
        public final Option<String> nameAlternative;
        public final ResourceType type;
        public final Email email;

        public AvailabilityIntervalResourceInfo(
                String name, Option<String> nameAlternative, ResourceType type, Email email)
        {
            this.name = name;
            this.nameAlternative = nameAlternative;
            this.type = type;
            this.email = email;
        }

        public static Function<ResourceInfo, AvailabilityIntervalResourceInfo> consF(final Language lang) {
            return new Function<ResourceInfo, AvailabilityIntervalResourceInfo>() {
                public AvailabilityIntervalResourceInfo apply(ResourceInfo r) {
                    return new AvailabilityIntervalResourceInfo(
                            r.getNameI18n(lang).getOrElse(""), r.getAlterNameI18n(lang),
                            r.getResource().getType(), r.getEmail());
                }
            };
        }
    }
}
