package ru.yandex.calendar.frontend.webNew.dto.out;

import lombok.Getter;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.email.Email;

/**
 * @author gutman
 */
public class SuggestInfo {

    @Getter
    @BenderBindAllFields
    public static class WithPlaces extends SuggestInfo {
        private final ListF<IntervalAndPlaces> intervals;
        private final Option<LocalDateTime> backwardSearchStart;
        private final Option<LocalDateTime> nextSearchStart;

        public WithPlaces(
                ListF<IntervalAndPlaces> intervals,
                Option<LocalDateTime> backwardSearchStart,
                Option<LocalDateTime> nextSearchStart)
        {
            this.intervals = intervals;
            this.backwardSearchStart = backwardSearchStart;
            this.nextSearchStart = nextSearchStart;
        }

        public Option<IntervalAndPlaces> findByStart(LocalDateTime start) {
            return intervals.find(i -> i.getInterval().getStart().isEqual(start));
        }

        public Option<ListF<Email>> findResourceEmailsByStart(LocalDateTime start) {
            return findByStart(start).map(IntervalAndPlaces::getResourceEmails);
        }
    }

    @Getter
    @BenderBindAllFields
    public static class WithNoPlaces extends SuggestInfo {
        private final ListF<IntervalAndOptions> intervals;
        private final Option<LocalDateTime> backwardSearchStart;
        private final Option<LocalDateTime> nextSearchStart;

        public WithNoPlaces(
                ListF<IntervalAndOptions> intervals,
                Option<LocalDateTime> backwardSearchStart,
                Option<LocalDateTime> nextSearchStart)
        {
            this.intervals = intervals;
            this.backwardSearchStart = backwardSearchStart;
            this.nextSearchStart = nextSearchStart;
        }

        public Option<IntervalWithDue> findByStart(LocalDateTime start) {
            return intervals.iterator().flatMap(i -> i.getOptions().iterator()).find(i -> i.getStart().isEqual(start));
        }
    }

    private SuggestInfo() {
    }

    public static WithPlaces withPlaces(
            ListF<IntervalAndPlaces> intervals,
            Option<LocalDateTime> backwardSearchStart, Option<LocalDateTime> nextSearchStart)
    {
        return new WithPlaces(intervals, backwardSearchStart, nextSearchStart);
    }

    public static WithNoPlaces withNoPlaces(
            ListF<IntervalAndOptions> intervals,
            Option<LocalDateTime> backwardSearchStart, Option<LocalDateTime> nextSearchStart)
    {
        return new WithNoPlaces(intervals, backwardSearchStart, nextSearchStart);
    }

    public static WithNoPlaces empty() {
        return new WithNoPlaces(Cf.list(), Option.empty(), Option.empty());
    }

    public WithPlaces asWithPlaces() {
        return (WithPlaces) this;
    }

    public WithNoPlaces asWithNoPlaces() {
        return (WithNoPlaces) this;
    }

    @Getter
    @Bendable
    public static class IntervalAndPlaces {
        @BenderPart
        @BenderFlatten
        private final LocalDateTimeInterval interval;
        @BenderPart(name = "place", wrapperName = "places")
        private final ListF<Place> places;
        @BenderPart
        private final boolean isPreferred;

        public IntervalAndPlaces(LocalDateTimeInterval interval, ListF<Place> places, boolean isPreferred) {
            this.interval = interval;
            this.places = places;
            this.isPreferred = isPreferred;
        }

        public ListF<Email> getResourceEmails() {
            return places.flatMap(p -> p.getResources().map(WebResourceInfo::getEmail));
        }

        public Option<WebResourceInfo> findResource(Email email) {
            return places.iterator().flatMap(p -> p.getResources().iterator()).find(r -> r.email.equalsIgnoreCase(email));
        }
    }

    @Getter
    @BenderBindAllFields
    public static class IntervalAndOptions {
        private final ListF<IntervalWithDue> options;
        private final boolean isPreferred;

        public IntervalAndOptions(ListF<IntervalWithDue> options, boolean isPreferred) {
            this.options = options;
            this.isPreferred = isPreferred;
        }
    }

    @Getter
    @BenderBindAllFields
    public static class IntervalWithDue {
        @BenderFlatten
        private final LocalDateTimeInterval interval;
        private final Option<LocalDate> dueDate;

        public IntervalWithDue(LocalDateTime start, LocalDateTime end, Option<LocalDate> dueDate) {
            this(new LocalDateTimeInterval(start, end), dueDate);
        }

        public IntervalWithDue(LocalDateTimeInterval interval, Option<LocalDate> dueDate) {
            this.interval = interval;
            this.dueDate = dueDate;
        }

        public LocalDateTime getStart() {
            return interval.getStart();
        }

        public LocalDateTime getEnd() {
            return interval.getEnd();
        }

        public Option<LocalDate> getDueDate() {
            return dueDate;
        }
    }

    @Getter
    @Bendable
    public static class Place {
        @BenderPart
        @BenderFlatten
        private final OfficeInfo office;
        @BenderPart
        @BenderFlatten
        private final LocalDateTimeInterval officeLocalIntervals;
        @BenderPart(name = "resource", wrapperName = "resources")
        private final ListF<WebResourceInfo> resources;
        @BenderPart
        private final boolean hasMoreFreeResources;

        public Place(
                OfficeInfo office, LocalDateTimeInterval officeLocalInterval,
                ListF<WebResourceInfo> resources, boolean hasMoreFreeResources)
        {
            this.office = office;
            this.officeLocalIntervals = officeLocalInterval;
            this.resources = resources;
            this.hasMoreFreeResources = hasMoreFreeResources;
        }
    }

    @Getter
    @Bendable
    public static class OfficeInfo {
        @BenderPart
        private final long officeId;
        @BenderPart
        private final String officeName;

        public OfficeInfo(long officeId, String officeName) {
            this.officeId = officeId;
            this.officeName = officeName;
        }
    }

    @Getter
    @Bendable
    public static class WebResourceInfo {
        @BenderPart
        private final long id;
        @BenderPart
        private final String name;
        @BenderPart
        private final Email email;
        @BenderPart
        private final ResourceType type;
        @BenderPart
        private final Option<LocalDate> dueDate;
        @BenderPart
        private final boolean hasPhone;
        @BenderPart
        private final boolean hasVideo;
        @BenderPart
        private final Option<Integer> floor;
        @BenderPart
        private final Option<String> group;
        @BenderPart
        private final Option<Boolean> canAdmin;

        public WebResourceInfo(
                long id, String name, Email email,
                ResourceType type, Option<LocalDate> dueDate,
                boolean hasPhone, boolean hasVideo,  Option<Integer> floor,
                Option<String> group, Option<Boolean> canAdmin)
        {
            this.id = id;
            this.name = name;
            this.email = email;
            this.type = type;
            this.dueDate = dueDate;
            this.hasPhone = hasPhone;
            this.hasVideo = hasVideo;
            this.floor = floor;
            this.group = group;
            this.canAdmin = canAdmin;
        }
    }

}
