package ru.yandex.calendar.frontend.webNew.dto.out;

import lombok.Getter;
import org.joda.time.LocalDate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.calendar.frontend.webNew.dto.inOut.NoNotificationsRange;
import ru.yandex.calendar.frontend.webNew.dto.inOut.TodoEmailTimes;
import ru.yandex.calendar.logic.beans.generated.Office;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.logic.beans.generated.SettingsYt;
import ru.yandex.calendar.logic.resource.OfficeManager;
import ru.yandex.calendar.logic.sharing.InvAcceptingType;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.util.dates.AuxDateTime;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.email.Email;

@Getter
@Bendable
@BenderMembersToBind(MembersToBind.ALL_FIELDS)
public class UserSettingsInfo {
    private final PassportUid uid;
    private final String login;
    private final Email email;
    private final String name;

    private final boolean isRoomAdmin;
    private final boolean isExternalUser;

    private final String tz;
    private final String lastOfferedGeoTz;

    private final boolean transliterateSms;

    private final NoNotificationsRange noNotificationsRange;
    private final TodoEmailTimes todoEmailTimes;

    private final String weekStartDay;
    private final int dayStartHour;
    private final String defaultView;

    private final boolean autoAcceptEventInvitations;
    private final boolean showTodosInGrid;
    private final boolean showAvailabilityToAnyone;

    private final boolean useYaMaps;
    private final boolean showShortForm;

    private final Option<Long> currentOfficeId;
    private final Option<String> currentOfficeTz;

    private final Option<Boolean> remindUndecided;
    private final Option<Boolean> letParticipantsEdit;
    private final Option<Boolean> noNotificationsInYaMail;
    private final Option<Boolean> noNotificationsDuringAbsence;

    private final Option<LocalDate> hiredAt;

    public UserSettingsInfo(
            SettingsInfo settings, UserInfo userInfo, Option<Office> currentOffice)
    {
        Settings common = settings.getCommon();
        Option<SettingsYt> yt = settings.getYt();

        this.uid = settings.getUid();
        this.login = settings.getUserLogin().getOrElse("");
        this.email = settings.getEmail();
        this.name = settings.getUserName().getOrElse("");

        this.isRoomAdmin = userInfo.isRoomAdmin();
        this.isExternalUser = userInfo.isExternalYt();

        this.tz = common.getTimezoneJavaid();
        this.lastOfferedGeoTz = settings.getCommon().getGeoTzJavaid();

        this.transliterateSms = common.getTranslitSms();

        this.noNotificationsRange = new NoNotificationsRange(
                common.getNoNtfStartTs().map(AuxDateTime.instantDateF(settings.getTz())),
                common.getNoNtfEndTs().map(AuxDateTime.instantDateF(settings.getTz())),
                common.getNoNtfStartTm().map(AuxDateTime.localTimeFromMillisOfDayF()),
                common.getNoNtfEndTm().map(AuxDateTime.localTimeFromMillisOfDayF()));

        this.todoEmailTimes = new TodoEmailTimes(
                common.getTodoPlannedEmailTm().map(AuxDateTime.localTimeFromMillisOfDayF()),
                common.getTodoExpiredEmailTm().map(AuxDateTime.localTimeFromMillisOfDayF()));

        this.weekStartDay = common.getStartWeekday().getDbValue();
        this.dayStartHour = common.getGridTopHours();
        this.defaultView = common.getViewType().toDbValue();

        this.autoAcceptEventInvitations = common.getInvAcceptType() != InvAcceptingType.MANUAL;
        this.showTodosInGrid = common.getShowTodo();
        this.showAvailabilityToAnyone = common.getShowAvailability();

        this.useYaMaps = common.getMapsEnabled();
        this.showShortForm = common.getShowShortForm().orElse(true);

        this.currentOfficeId = currentOffice.map(Office.getIdF());
        this.currentOfficeTz = currentOffice.map(OfficeManager.getOfficeTimeZoneF().andThen(AuxDateTime.getTzIdF()));

        this.remindUndecided = yt.map(SettingsYt::getRemindUndecided);
        this.letParticipantsEdit = yt.map(SettingsYt.getLetParticipantsEditF());
        this.noNotificationsInYaMail = yt.map(Function1B.wrap(SettingsYt.getXivaReminderEnabledF()).notF().asFunction());
        this.noNotificationsDuringAbsence = yt.map(Function1B.wrap(SettingsYt.getNoNtfDuringAbsenceF()).asFunction());

        var hiredInstant = settings.getCommon().getCreationTs();
        if (hiredInstant.isPresent()) {
            var hiredLocalDate = hiredInstant.get().toDateTime().toLocalDate();
            this.hiredAt = Option.of(hiredLocalDate);
        } else {
            this.hiredAt = Option.empty();
        }
    }

}
