package ru.yandex.calendar.frontend.webNew.dto.out;

import lombok.Value;
import lombok.With;
import lombok.val;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDateTime;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.bender.RawJson;
import ru.yandex.calendar.frontend.bender.WebDateTime;
import ru.yandex.calendar.frontend.webNew.dto.inOut.EventAttachmentData;
import ru.yandex.calendar.frontend.webNew.dto.inOut.RepetitionData;
import ru.yandex.calendar.logic.beans.generated.Event;
import ru.yandex.calendar.logic.beans.generated.Repetition;
import ru.yandex.calendar.logic.event.EventActions;
import ru.yandex.calendar.logic.event.EventInterval;
import ru.yandex.calendar.logic.event.EventRoutines;
import ru.yandex.calendar.logic.event.avail.Availability;
import ru.yandex.calendar.logic.event.model.EventType;
import ru.yandex.calendar.logic.notification.Notification;
import ru.yandex.calendar.logic.sharing.Decision;
import ru.yandex.calendar.logic.sharing.perm.EventActionClass;
import ru.yandex.calendar.logic.sharing.perm.EventInfoForPermsCheck;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;

@Value
@Bendable
public class WebEventInfo {
    @BenderPart
    long id;
    @BenderPart
    String externalId;
    @BenderPart
    int sequence;
    @BenderPart
    EventType type;
    @BenderPart
    String name;
    @BenderPart
    String description;
    @BenderPart(name = "attachment", wrapperName = "attachments")
    Option<ListF<EventAttachmentData>> attachments;
    @BenderPart
    String location;
    @BenderPart
    String descriptionHtml;
    @BenderPart
    String locationHtml;
    @BenderPart
    WebDateTime startTs;
    @BenderPart
    WebDateTime endTs;
    @BenderPart
    LocalDateTime instanceStartTs;
    @BenderPart
    boolean isAllDay;
    @BenderPart
    boolean isRecurrence;
    @BenderPart
    Option<WebUserParticipantInfo> organizer;
    @BenderPart(name = "attendee", wrapperName = "attendees")
    @With
    ListF<WebUserParticipantInfo> attendees;
    @BenderPart(name = "optionalAttendee", wrapperName = "optionalAttendees")
    @With
    ListF<WebUserParticipantInfo> optionalAttendees;
    @BenderPart(name = "resource", wrapperName = "resources")
    ListF<WebResourceInfo> resources;
    @BenderPart
    ListF<WebUserInfo> subscribers;
    @BenderPart(name = "notification", wrapperName = "notifications")
    ListF<Notification> notifications;
    @BenderPart
    Option<RepetitionData> repetition;
    @BenderPart
    EventActions actions;
    @BenderPart
    boolean participantsCanInvite;
    @BenderPart
    boolean participantsCanEdit;
    @BenderPart
    boolean othersCanView;
    @BenderPart
    Option<Decision> decision;
    @BenderPart
    Option<Availability> availability;
    @BenderPart
    Option<Long> layerId;
    @BenderPart
    boolean organizerLetToEditAnyMeeting;
    @BenderPart
    boolean canAdminAllResources;
    @BenderPart
    boolean repetitionNeedsConfirmation;
    @BenderPart
    int totalAttendees;
    @BenderPart
    int totalOptionalAttendees;
    @BenderPart
    Option<RawJson> data;
    @BenderPart
    String conferenceUrl;

    public static WebEventInfo create(
            EventInterval interval,
            String externalId,
            Event event,
            Option<ListF<EventAttachmentData>> attachments,
            EventInfoForPermsCheck permsInfo,
            Option<Repetition> repetition,
            boolean repetitionNeedsConfirmation,
            Option<WebUserParticipantInfo> organizer,
            ListF<WebResourceInfo> resources,
            ListF<WebUserParticipantInfo> attendees,
            ListF<WebUserParticipantInfo> optionalAttendees,
            ListF<WebUserInfo> subscribers,
            ListF<Notification> notifications,
            EventActions actions,
            Option<Decision> decision,
            Option<Availability> availability,
            Option<Long> layerId,
            DateTimeZone tz,
            Option<String> linkSignKey,
            boolean canAdminAllResources) {
        val id = event.getId();
        val sequence = event.getSequence();
        val type = event.getType();
        val name = event.getName();
        val description = event.getDescription();
        val location = event.getLocation();
        val descriptionHtml = EventRoutines.getDescriptionHtmlForVerstka(event.getDescription(), linkSignKey);
        val locationHtml = EventRoutines.getLocationHtmlForVerstka(event.getLocation(), linkSignKey);
        val startTs = WebDateTime.dateTime(interval.getStart().toDateTime(tz));
        val endTs = WebDateTime.dateTime(interval.getEnd().toDateTime(tz));
        val instanceStartTs = new LocalDateTime(interval.getInstanceStart(), DateTimeZone.UTC);
        val isAllDay = event.getIsAllDay();
        val isRecurrence = event.getRecurrenceId().isPresent();
        Option<RepetitionData> repetitionData = repetition.isPresent()
                ? Option.of(RepetitionData.fromRepetition(repetition.get(), tz))
                : Option.empty();
        val participantsCanInvite = event.getParticipantsInvite();
        val participantsCanEdit = event.getPermParticipants() == EventActionClass.EDIT;
        val othersCanView = event.getPermAll() == EventActionClass.VIEW;
        val organizerLetToEditAnyMeeting = permsInfo.isOrganizerLetToEditAnyMeeting();
        val totalAttendees = attendees.length();
        val totalOptionalAttendees = optionalAttendees.length();
        val conferenceUrl = event.getConferenceUrl().orElse("");

        return new WebEventInfo(id, externalId, sequence, type, name, description, attachments, location, descriptionHtml, locationHtml,
                startTs, endTs, instanceStartTs, isAllDay, isRecurrence, organizer, attendees, optionalAttendees, resources,
                subscribers, notifications, repetitionData, actions, participantsCanInvite, participantsCanEdit,
                othersCanView, decision, availability, layerId, organizerLetToEditAnyMeeting, canAdminAllResources,
                repetitionNeedsConfirmation, totalAttendees, totalOptionalAttendees, event.getData(), conferenceUrl);
    }
}
