package ru.yandex.calendar.frontend.worker;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.boot.CalendarBazingaClientContextConfiguration;
import ru.yandex.calendar.frontend.worker.task.HandleMailAttachTask;
import ru.yandex.calendar.frontend.worker.task.SendMailTask;
import ru.yandex.calendar.frontend.worker.task.StaffSynchronizerTask;
import ru.yandex.calendar.frontend.worker.task.UpdateIcsFeedTask;
import ru.yandex.calendar.zk.CalendarZkContextConfiguration;
import ru.yandex.commune.actor.typed.dynamic.DynamicConfigurationTypedServer;
import ru.yandex.commune.bazinga.BazingaActorServerHolder;
import ru.yandex.commune.bazinga.BazingaConfiguration;
import ru.yandex.commune.bazinga.admin.BazingaAdminAddressResolver;
import ru.yandex.commune.bazinga.admin.BazingaAdminAddressResolverStandard;
import ru.yandex.commune.bazinga.context.BazingaAdminContextConfiguration;
import ru.yandex.commune.bazinga.context.BazingaHeartbeatContextConfiguration;
import ru.yandex.commune.bazinga.context.BazingaMonitoringContextConfiguration;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.bazinga.impl.worker.BazingaWorkerConfiguration;
import ru.yandex.commune.bazinga.impl.worker.OnetimeTaskLogConfiguration;
import ru.yandex.commune.bazinga.impl.worker.TaskExecutorMetrics;
import ru.yandex.commune.bazinga.impl.worker.WorkerLogConfiguration;
import ru.yandex.commune.bazinga.pg.context.PgBazingaMaintenanceContextConfiguration;
import ru.yandex.commune.bazinga.pg.context.PgBazingaWorkerContextConfiguration;
import ru.yandex.commune.bazinga.pg.fetcher.FetchLimits;
import ru.yandex.commune.bazinga.pg.fetcher.PgBazingaJobsFetcherConfiguration;
import ru.yandex.commune.bazinga.pg.storage.maintenance.TaskCleanupSettings;
import ru.yandex.commune.bazinga.pg.storage.maintenance.TaskCleanupSettingsProvider;
import ru.yandex.commune.bazinga.scheduler.TaskCategory;
import ru.yandex.commune.bazinga.scheduler.TaskQueue;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.ip.InternetDomainName;
import ru.yandex.misc.ip.IpPort;
import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.misc.property.PropertiesHolder;
import ru.yandex.misc.time.TimeUtils;

@Import({
        PgBazingaWorkerContextConfiguration.class,
        CalendarBazingaControllerContextConfiguration.class,
        BazingaAdminContextConfiguration.class,
        BazingaHeartbeatContextConfiguration.class,
        BazingaMonitoringContextConfiguration.class,
        PgBazingaMaintenanceContextConfiguration.class,
        CalendarBazingaClientContextConfiguration.class,
        CalendarZkContextConfiguration.class,
})
@Configuration
public class CalendarBazingaContextConfiguration {

    @Value("${bazinga.port}")
    private int bazingaPort;

    @Autowired
    private ZkPath zkRoot;

    @Autowired
    private EnvironmentType environmentType;

    private ZkPath bazingaZkPath() {
        return zkRoot.child("bazinga");
    }

    @Bean
    public BazingaConfiguration bazingaConfiguration() {
        return new BazingaConfiguration(
                bazingaZkPath(),
                new InternetDomainName(HostnameUtils.localHostname()),
                IpPort.cons(bazingaPort)
        );
    }

    @Bean
    public BazingaWorkerConfiguration bazingaWorkerConfiguration(
            @Value("${bazinga.worker.cron.threads}") int cronThreads,
            @Value("${bazinga.worker.cron.queue}") int cronQueue,

            @Value("${bazinga.worker.onetime.threads}") int onetimeThreads,
            @Value("${bazinga.worker.onetime.queue}") int onetimeQueue,

            @Value("${bazinga.worker.ics-feeds.threads}") int icsFeedThreads,
            @Value("${bazinga.worker.ics-feeds.queue}") int icsFeedsQueue,

            @Value("${bazinga.worker.mailhook.threads}") int mailhookThreads,
            @Value("${bazinga.worker.mailhook.queue}") int mailhookQueue,

            @Value("${bazinga.worker.sendmail.threads}") int sendmailThreads,
            @Value("${bazinga.worker.sendmail.queue}") int sendmailQueue,

            @Value("${bazinga.log.cron.dir.size.limit.mb}") int cronLogDirSizeLimitMb,
            @Value("${bazinga.log.cron.dir.count.limit}") int cronLogDirFilesCountLimit,
            @Value("${bazinga.log.cron.task.size.limit.mb}") int cronTaskLogSizeLimitMb,
            @Value("${bazinga.log.cron.task.count.limit}") int cronTaskLogsCountLimit,

            @Value("${bazinga.log.onetime.sizeLimit.mb}") int onetimeLogFileSizeLimitMb,
            @Value("${bazinga.log.onetime.historyCount}") int onetimeLogFileHistoryCount,
            @Value("${bazinga.log.onetime.tooOldTimeout.hours}") int onetimeLogTooOldTimeoutHours,

            @Value("${bazinga.worker.staff-sync.threads}") int staffSyncThreads) {
        return new BazingaWorkerConfiguration(
                Cf.list(TaskCategory.DEFAULT),
                new WorkerLogConfiguration(
                        new File2("/var/log/calendar/bazinga"),
                        Option.of(DataSize.fromMegaBytes(cronLogDirSizeLimitMb)),
                        Option.of(cronLogDirFilesCountLimit),
                        Option.of(DataSize.fromMegaBytes(cronTaskLogSizeLimitMb)),
                        Option.of(cronTaskLogsCountLimit),
                        Option.of(new OnetimeTaskLogConfiguration(
                                DataSize.fromMegaBytes(onetimeLogFileSizeLimitMb),
                                onetimeLogFileHistoryCount,
                                Duration.standardHours(onetimeLogTooOldTimeoutHours),
                                new Duration(0)))),
                Option.empty(),
                Cf.list(
                        new TaskQueue(TaskQueueName.CRON, cronThreads, cronQueue),
                        new TaskQueue(TaskQueueName.REGULAR, onetimeThreads, onetimeQueue),
                        new TaskQueue(HandleMailAttachTask.QUEUE_NAME, mailhookThreads, mailhookQueue),
                        new TaskQueue(UpdateIcsFeedTask.QUEUE_NAME, icsFeedThreads, icsFeedsQueue),
                        new TaskQueue(SendMailTask.QUEUE_NAME, sendmailThreads, sendmailQueue),
                        new TaskQueue(StaffSynchronizerTask.QUEUE_NAME, staffSyncThreads, cronQueue)
                ));
    }

    @Bean
    public PgBazingaJobsFetcherConfiguration pgBazingaJobsFetcherConfiguration(
            @Value("${bazinga.fetch.delay.millis}") int delayMs,
            @Value("${bazinga.fetch.threads}") int threads,
            @Value("${bazinga.fetch.limits.taskMin}") int taskMinLimit,
            @Value("${bazinga.fetch.limits.total}") int totalLimit) {
        return new PgBazingaJobsFetcherConfiguration(
                bazingaConfiguration().getHostPort(), Duration.millis(delayMs),
                threads, () -> new FetchLimits(taskMinLimit, totalLimit));
    }

    @Bean
    public TaskCleanupSettingsProvider cleanupSettingsProvider() {
        MapF<TaskId, Duration> completedTtl = Cf.hashMap();
        MapF<TaskId, Duration> failedTtl = Cf.hashMap();

        PropertiesHolder.properties().forEach((k, v) -> {
            if (k instanceof String && ((String) k).startsWith("bazinga.tasks-ttl.")) {
                ListF<String> parts = Cf.x(((String) k).split("\\."));

                TaskId taskId = new TaskId(String.join(".", parts.subList(2, parts.length() - 1)));

                MapF<TaskId, Duration> target = parts.last().equals("completed") ? completedTtl
                        : parts.last().equals("failed") ? failedTtl : Cf.map();

                target.put(taskId, TimeUtils.toDuration((String) v));
            }
        });
        TaskId defaultTaskId = new TaskId("default");

        return taskId -> new TaskCleanupSettings(taskId,
                completedTtl.getOrElse(taskId, completedTtl.getOrThrow(defaultTaskId)),
                failedTtl.getOrElse(taskId, failedTtl.getOrThrow(defaultTaskId)));
    }

    @Bean
    public BazingaActorServerHolder bazingaActorServerHolder() {
        return new BazingaActorServerHolder(typedServer());
    }

    @Bean
    public DynamicConfigurationTypedServer typedServer() {
        return new DynamicConfigurationTypedServer(IpPort.cons(bazingaPort));
    }

    @Bean
    public BazingaAdminAddressResolver bazingaAdminAddressResolver() {
        return new BazingaAdminAddressResolverStandard("calendar-worker");
    }

    @Bean
    public TaskExecutorMetrics taskExecutorMetrics() {
        return new UnistatWorkerTaskExecutionMetrics();
    }

    @Bean
    public WorkerMetrics workerMetrics() {
        return new WorkerMetrics();
    }
}
