package ru.yandex.calendar.frontend.worker.task;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.worker.CalendarCronTask;
import ru.yandex.calendar.logic.beans.generated.EventUser;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.event.EventAttachedLayerId;
import ru.yandex.calendar.logic.event.EventInstanceInfo;
import ru.yandex.calendar.logic.event.EventInvitationManager;
import ru.yandex.calendar.logic.event.web.EventWebManager;
import ru.yandex.calendar.logic.notification.NotificationsData;
import ru.yandex.calendar.logic.user.Group;
import ru.yandex.calendar.logic.user.UserInfo;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.SchedulePeriodic;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.email.Email;

/**
 * Task for periodically transferring all participants to subscribers,
 * except for the specified ones.
 *
 * @author Stepan Kladikov
 */

@Slf4j
public class AttendeeToSubscriberTask extends CalendarCronTask {

    @Autowired
    private UserManager userManager;
    @Autowired
    private EventInvitationManager eventInvitationManager;
    @Autowired
    private EventWebManager eventWebManager;

    private UserInfo rootUser;
    private static final String RID_TAG = "req-id-script-replace-attendee-by-subscriber";

    private final DynamicProperty<Long> eventIdProperty = new DynamicProperty<>(
            "tmp.attendee-to-subscriber-task-event-id", -1L);
    private final DynamicProperty<ListF<Long>> userIdsProperty = new DynamicProperty<>(
            "tmp.attendee-to-subscriber-task-user-ids", Cf.list());

    @PostConstruct
    private void initRootUser() {
        Email rootUserEmail = new Email("robot-calendar@yandex-team.ru");
        PassportUid rootUserUid = userManager.getUidByEmail(rootUserEmail).get();
        rootUser = userManager.getUserInfo(rootUserUid);
    }

    @Override
    public Schedule cronExpression() {
        return new SchedulePeriodic(Duration.standardMinutes(10));
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(30);
    }

    @Override
    public void doExecute(ExecutionContext executionContext) throws Exception {
        AttendeeToSubscriberTaskResult result = new AttendeeToSubscriberTaskResult();

        Long eventId = eventIdProperty.get();
        List<PassportUid> userIds = userIdsProperty.get().stream()
                .map(PassportUid::cons)
                .collect(Collectors.toList());

        if (eventId == -1L || userIds.isEmpty()) {
            log.info("Data is inconsistent");
            executionContext.setExecutionInfo(result);
            return;
        }


        for (PassportUid userId : userIds) {
            Email email = userManager.getEmailByUid(userId).get();
            UserData user = new UserData(userId, email);

            if (isSubscriber(user, eventId)) {
                log.info("Subscription for user {} already exist", user.uid.getUid());
                result.registerSkipped();
            } else {
                boolean isSuccess = move(eventId, user);
                result.registerResult(isSuccess);
            }
        }

        executionContext.setExecutionInfo(result);
    }

    private boolean isSubscriber(UserData user, Long eventId) {
        ActionInfo actionInfo = new ActionInfo(ActionSource.PERFORMER, RID_TAG, Instant.now());

        EventInstanceInfo eventInfo = eventWebManager
                .getEvent(Option.of(user.uid), eventId, Option.empty(), Option.empty(), actionInfo);
        Option<EventUser> eventUser = eventInfo.getEventUser();

        return eventUser.isPresent() && eventUser.get().getIsSubscriber() && !eventUser.get().getIsAttendee();
    }

    private boolean move(Long eventId, UserData user) {
        try {
            removeAttendeeIfExist(eventId, user);
            createSubscription(eventId, user);
        } catch (Throwable e) {
            log.error("Undeclared error!", e);
            return false;
        }

        return true;
    }

    private void removeAttendeeIfExist(Long eventId, UserData user) {
        ActionInfo actionInfo = new ActionInfo(ActionSource.PERFORMER, RID_TAG, Instant.now());

        Option<EventAttachedLayerId> removingResult = eventInvitationManager
                .removeAttendeeByEmail(eventId, rootUser, user.email, actionInfo);

        if (removingResult.isEmpty()) {
            log.info("User {} was already removed from event", user.uid.getUid());
        }
    }

    private void createSubscription(Long eventId, UserData user) {
        ActionInfo actionInfo = new ActionInfo(ActionSource.PERFORMER, RID_TAG, Instant.now());
        EventUser eventUser = new EventUser();
        eventUser.setIsSubscriber(true);
        UserInfo userInfo = userManager.getUserInfo(user.uid);
        UserInfo superUserInfo = new UserInfo(new PassportUid(userInfo.getUid().getUid()), java.util.EnumSet.of(Group.SUPER_USER), Cf.list(), Cf.list(), false, false);

        eventWebManager.attachEvent(superUserInfo, eventId, Option.empty(), eventUser, new NotificationsData.UseLayerDefaultIfCreate(), actionInfo);
    }

    private class UserData {
        public PassportUid uid;
        public Email email;

        UserData(PassportUid uid, Email email) {
            this.uid = uid;
            this.email = email;
        }
    }
}

