package ru.yandex.calendar.frontend.worker.task;

import java.util.Optional;

import lombok.Data;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.webNew.dto.in.ImportIcsData;
import ru.yandex.calendar.frontend.worker.CalendarOnetimeTask;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.ics.feed.IcsFeedManager;
import ru.yandex.calendar.logic.ics.imp.IcsImportMode;
import ru.yandex.calendar.logic.ics.imp.IcsImporter;
import ru.yandex.calendar.logic.ics.imp.LayerReference;
import ru.yandex.calendar.logic.ics.iv5j.ical.IcsCalendar;
import ru.yandex.calendar.logic.sending.bazinga.MessageExtraDao;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleLinear;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

@Slf4j
public class IcsImportTask extends CalendarOnetimeTask<IcsImportTask.Parameters> {
    @Autowired
    private IcsImporter icsImporter;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;
    @Autowired
    private IcsFeedManager icsFeedManager;
    @Autowired
    private MessageExtraDao messageExtraDao;

    public IcsImportTask(PassportUid uid, long messageExtraId, long layerId) {
        super(new Parameters(uid, messageExtraId, layerId));
    }

    public IcsImportTask() {
        super(Parameters.class);
    }

    @Override
    protected ActionSource getActionSource() {
        return ActionSource.WEB_ICS;
    }

    @SneakyThrows
    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) throws Exception {
        try {
            IcsCalendar ics = parameters.messageExtraId
                    .map(id -> messageExtraDao.load(id).get().properties)
                    .or(() -> ImportIcsData.getIcs(parameters.ics))
                    .or(() -> parameters.url.map(url -> icsFeedManager.downloadAndParseNow(url)))
                    .orElseThrow();

            icsImporter.importIcsStuff(parameters.passportUid, ics, IcsImportMode.importFile(LayerReference.id(parameters.layerId)), parameters.getUrl());
        } finally {
            val actionInfo = CalendarRequest.getCurrent().getActionInfo();
            xivaNotificationManager.notifyLayersUsersAboutEventsChange(Cf.list(parameters.layerId), actionInfo);
        }
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_FINISHED, ActiveUidDuplicateBehavior.MERGE_IF_READY);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleLinear(Duration.standardMinutes(1), 1);
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(30);
    }

    @Override
    public int priority() {
        return 0;
    }

    @Data
    @BenderBindAllFields
    public static class Parameters {
        private final PassportUid passportUid;
        private final Optional<Long> messageExtraId;
        private final Optional<ImportIcsData.IcsWrapper> ics;
        private final Optional<String> url;
        private final long layerId;

        public Parameters(PassportUid passportUid, long messageExtraId, long layerId) {
            this.passportUid = passportUid;
            this.messageExtraId = Optional.of(messageExtraId);
            this.ics = Optional.empty();
            this.url = Optional.empty();
            this.layerId = layerId;
        }
    }
}
