package ru.yandex.calendar.frontend.worker.task;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.UUID;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.worker.CalendarOnetimeTask;
import ru.yandex.calendar.logic.beans.generated.SendingMail;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.sending.bazinga.MessageExtraDao;
import ru.yandex.calendar.logic.sending.bazinga.SendingMailDao;
import ru.yandex.calendar.logic.sending.param.MessageDestination;
import ru.yandex.calendar.logic.sending.param.MessageOverrides;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.real.DestinedMessage;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.sharing.MailType;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author dbrylev
 */
@Slf4j
public class SendMailTask extends CalendarOnetimeTask<SendMailTask.Parameters> {
    public static final TaskQueueName QUEUE_NAME = new TaskQueueName("sendmail");

    @Autowired
    private MailSender sender;
    @Autowired
    private SendingMailDao sendingMailDao;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private MessageExtraDao messageExtraDao;

    private boolean has450 = false;
    private boolean noSuchUser = false;

    public SendMailTask() {
        super(SendMailTask.Parameters.class);
    }

    public SendMailTask(DestinedMessage destined, ActionInfo actionInfo) {
        super(new Parameters(
                destined.getXsl(), destined.getDestination(),
                actionInfo.getActionSource(), actionInfo.getRequestIdWithHostId()));
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) throws Exception {
        UUID jobId = context.getFullJobId().getJobId().getUuid();

        Option<SendingMail> mail = sendingMailDao.findByJobId(jobId).singleO();

        if (!mail.isPresent()) {
            return;
        }

        MessageOverrides overrides = mail.get().getParameters().getMessageOverrides();

        MessageParameters message = mail.get().getParameters().withOverrides(overrides
                .withDate(overrides.date.getOrElse(mail.get().getCreationTs()))
                .withMessageId(overrides.messageId.getOrElse(jobId.toString() + "@" + svcRoutines.getSelfUrlHost())));

        message.loadExtra(messageExtraDao);

        transactionTemplate.execute(s -> {
            if (sendingMailDao.deleteByJobId(jobId) > 0) {
                try {
                    sender.send(new DestinedMessage(parameters.xsl, parameters.destination, message),
                            new ActionInfo("sendMailTask", parameters.source, parameters.rid + "/" + jobId,
                                    Instant.now()));
                } catch (Exception e) {
                    boolean isRetryable = true;
                    if (is450ResponseCode(e)) {
                        if (message.mailType() == MailType.EVENT_NOTIFICATION) {
                            isRetryable = false;
                        }
                        has450 = true;
                    }
                    if (is550ResponseCode(e)){
                        // do not retry if user does not exist
                        isRetryable = false;
                        noSuchUser = true;
                    }
                    log.error("task failed, has_450: " + has450
                            + ", no_such_user = " + noSuchUser
                            + ", is_retryable = " + isRetryable
                            + ", mailType:" + message.mailType()
                            + ", recipient:" + message.getRecipient()
                    );
                    if (isRetryable) {
                        throw e;
                    }
                }
            }
            return null;
        });
    }

    private boolean is450ResponseCode(Exception e) {
        final StringWriter stringWriter = new StringWriter();
        e.printStackTrace(new PrintWriter(stringWriter));
        return stringWriter.toString().contains("450 4.2.1 The recipient has exceeded message rate limit");
    }

    private boolean is550ResponseCode(Exception e) {
        final StringWriter stringWriter = new StringWriter();
        e.printStackTrace(new PrintWriter(stringWriter));
        return stringWriter.toString().contains("550 5.7.1 No such user");
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        if (has450) {
            return new CompoundReschedulePolicy(
                    new RescheduleConstant(Duration.ZERO, 2),
                    new RescheduleConstant(Duration.standardSeconds(20), 50));
        } else {
            return new CompoundReschedulePolicy(
                    new RescheduleConstant(Duration.ZERO, 2),
                    new RescheduleExponential(Duration.standardMinutes(1), 10));
        }
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(1);
    }

    @Override
    public TaskQueueName queueName() {
        return QUEUE_NAME;
    }

    @Data
    @BenderBindAllFields
    public static class Parameters {
        private final DestinedMessage.XslSource xsl;
        private final MessageDestination destination;

        private final ActionSource source;
        private final String rid;
    }
}
