package ru.yandex.calendar.frontend.worker.task;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.frontend.worker.CalendarCronTask;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.resource.ResourceType;
import ru.yandex.calendar.logic.sending.real.ScheduleMailSender;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.time.TimeUtils;


@Slf4j
public class SendMassageScheduleTask extends CalendarCronTask {

    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    protected UserManager userManager;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private ScheduleMailSender scheduleMailSender;
    @Value("${ml.massage:-}")
    private String recipient;
    @Value("${ml.cron-expression:-}")
    private String cronExpression;

    public static final TaskQueueName QUEUE_NAME = new TaskQueueName("massage-schedule");

    @Override
    public Schedule cronExpression() {
        return ScheduleCron.parse(cronExpression, TimeUtils.EUROPE_MOSCOW_TIME_ZONE);
    }

    @Override
    public void doExecute(ExecutionContext executionContext) throws Exception {

        if (passportAuthDomainsHolder.containsYandexTeamRu()) {
            LocalDate date = LocalDate.now(DateTimeZone.UTC).plusDays(1);

            log.info("Start getting massage rooms schedule for " + DateTimeFormat.forPattern("dd.MM.yyyy").print(date));

            ListF<ResourceInfo> availableResources = resourceRoutines
                    .findActiveResources(Cf.list(ResourceType.MASSAGE_ROOM))
                    .map(Resource.getIdF().andThen(resourceRoutines.getResourceInfoByIdF()));

            ListF<String> scheduleDocuments = availableResources.map(resourceInfo -> scheduleMailSender.scheduleHtml(resourceInfo, date));
            log.info("Massage rooms schedule received for " + availableResources.size() + " rooms.");

            scheduleMailSender.sendSchedule(availableResources, scheduleDocuments, new Email(recipient), date);

            log.info("Massage rooms schedule sent successfully");
        }
    }

}
