package ru.yandex.calendar.frontend.worker.task;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.worker.CalendarOnetimeTask;
import ru.yandex.calendar.logic.beans.generated.TodoItemFields;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.sending.param.TodoMessageParameters;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.todo.TodoDao;
import ru.yandex.calendar.logic.todo.TodoListEmailManager;
import ru.yandex.calendar.logic.todo.TodoMailType;
import ru.yandex.calendar.logic.update.LockResource;
import ru.yandex.calendar.logic.update.PgTransactionLocker;
import ru.yandex.calendar.logic.user.SettingsInfo;
import ru.yandex.calendar.logic.user.SettingsRoutines;
import ru.yandex.calendar.util.db.DbUtils;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.log.reqid.RequestIdStack;

/**
 * @author dbrylev
 */
public class SendTodoMailTask extends CalendarOnetimeTask<SendTodoMailTask.Parameters> {

    @Autowired
    private PgBazingaStorage bazingaStorage;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private PgTransactionLocker pgTransactionLocker;
    @Autowired
    private SettingsRoutines settingsRoutines;
    @Autowired
    private TodoListEmailManager todoListEmailManager;
    @Autowired
    private MailSender mailSender;
    @Autowired
    private TodoDao todoDao;

    public SendTodoMailTask(PassportUid uid, TodoMailType type) {
        super(new Parameters(uid, type));
    }

    public SendTodoMailTask() {
        super(Parameters.class);
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) throws Exception {
        PassportUid uid = parameters.uid;

        ActionInfo actionInfo = new ActionInfo(ActionSource.WORKER, RequestIdStack.current().get(), Instant.now());

        Option<TodoMessageParameters> mail = todoListEmailManager.prepareTodoMail(uid, parameters.type, actionInfo);

        transactionTemplate.execute(DbUtils.transactionCallback(() -> {
            mailSender.sendEmailsViaTask(mail, actionInfo);
            settingsRoutines.invalidateCacheForUid(uid);

            pgTransactionLocker.lock(Cf.list(LockResource.settingsUser(uid)));
            SettingsInfo settings = settingsRoutines.getSettingsByUid(uid);

            bazingaStorage.deleteOnetimeJob(context.getFullJobId());

            pgTransactionLocker.lock(Cf.list(LockResource.todoUser(uid)));
            boolean hasMore = todoDao.getUserNotDeletedNotArchivedTodoItemsCount(uid, SqlCondition.trueCondition()
                    .and(TodoItemFields.COMPLETION_TS.column().isNull())
                    .and(TodoItemFields.DUE_TS.column().isNotNull())) > 0;

            if (hasMore) {
                todoListEmailManager.scheduleTodoMailTask(settings, parameters.type, actionInfo.getNow());
            }
        }));
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_FINISHED, ActiveUidDuplicateBehavior.MERGE_IF_READY);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleExponential(Duration.standardMinutes(1), 9);
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(2);
    }

    @Override
    public int priority() {
        return 0;
    }

    @BenderBindAllFields
    public static class Parameters {
        public final PassportUid uid;
        public final TodoMailType type;

        public Parameters(PassportUid uid, TodoMailType type) {
            this.uid = uid;
            this.type = type;
        }
    }
}
