package ru.yandex.calendar.frontend.worker.task;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.frontend.worker.CalendarCronTask;
import ru.yandex.calendar.logic.beans.generated.Resource;
import ru.yandex.calendar.logic.beans.generated.ResourceFields;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.resource.ResourceDao;
import ru.yandex.calendar.logic.resource.ResourceInfo;
import ru.yandex.calendar.logic.sending.param.BrokenDisplayReportMessageParameters;
import ru.yandex.calendar.logic.sending.param.MessageParameters;
import ru.yandex.calendar.logic.sending.real.MailSender;
import ru.yandex.calendar.logic.svc.SvcRoutines;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.resources.UStringLiteral;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.SchedulePeriodic;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.Zk;
import ru.yandex.commune.zk2.primitives.observer.ZkPathObserver;
import ru.yandex.inside.passport.blackbox.PassportAuthDomain;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.log.reqid.RequestIdStack;

/**
 * @author dbrylev
 */
public class TaskReportBrokenDisplays extends CalendarCronTask {

    @Autowired
    private ZkPathObserver zkObserver;
    @Autowired
    private ZkPath calendarZkPath;
    @Autowired
    private MailSender mailSender;
    @Autowired
    private SvcRoutines svcRoutines;
    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;

    private final DynamicProperty<Integer> displayBrokenAfterSec = new DynamicProperty<>("displayBrokenAfterSec", 180);

    private final DynamicProperty<ListF<String>> displayBrokenReportSubscribers = new DynamicProperty<>(
            "displayBrokenReportSubscribers", EnvironmentType.PRODUCTION == EnvironmentType.getActive()
                    ? Cf.list("*=broken-pads@", "redrose;rrs;rrm=help@")
                    : Cf.<String>list());

    @Override
    public void doExecute(ExecutionContext executionContext) throws Exception {
        if (!passportAuthDomainsHolder.containsYandexTeamRu()) return;

        ActionInfo actionInfo = new ActionInfo(
                ActionSource.WORKER, RequestIdStack.current().getOrThrow("no request id!"), Instant.now());

        Email senderEmail = svcRoutines.getCalendarInfoEmail(PassportAuthDomain.YANDEX_TEAM_RU);
        MailAddress sender = new MailAddress(senderEmail, UStringLiteral.YA_CALENDAR);

        Instant brokenPingBefore = actionInfo.getNow().minus(Duration.standardSeconds(displayBrokenAfterSec.get()));

        Zk zk = zkObserver.zkO().getOrThrow("Zk unavailable");

        Option<byte[]> previousData = zk.getDataIfExists(zkPath());
        String previousValue = CharsetUtils.decodeUtf8(previousData.getOrElse(new byte[0]));
        ListF<Long> previousIds = Cf.x(previousValue.split(",")).filterMap(Cf.Long.parseSafeF());

        ListF<Resource> brokenResources = resourceDao.findResources(ResourceFields.DISPLAY_TOKEN.column().isNotNull()
                .and(ResourceFields.DISPLAY_LAST_PING_TS.column().lt(brokenPingBefore)));
        ListF<Long> brokenIds = brokenResources.map(Resource.getIdF());

        if (previousIds.unique().equals(brokenIds.unique())) return;

        ListF<Long> recentlyBrokenIds = brokenIds.filter(previousIds.containsF().notF());
        ListF<ResourceInfo> resources = resourceDao.findResourceInfosByIds(recentlyBrokenIds);

        MapF<String, ListF<MailAddress>> recipientsByOfficeAbbr = displayBrokenReportSubscribers.get()
                .flatMap(subs -> Cf2.join(
                        Cf.x(subs.split("=")[0].split(";")),
                        Cf.x(subs.split("=")[1].split(";"))
                                .filterMap(e -> Email.parseSafe(e.endsWith("@") ? e + "yandex-team.ru" : e))))
                .groupByMapValues(Tuple2::get1, t -> new MailAddress(t.get2()));

        ListF<MessageParameters> mails = resources
                .flatMap(resource -> resource.getOffice().getAbbr().plus1("*")
                        .flatMap(abbr -> recipientsByOfficeAbbr.getOrElse(abbr, Cf.list()))
                        .stableUniqueBy(MailAddress::getEmail)
                        .map(recipient -> new BrokenDisplayReportMessageParameters(sender, recipient, resource)));

        if (!previousData.isPresent()) {
            zk.mkdirs(zkPath());
        }
        mailSender.sendEmailsViaTask(mails, actionInfo);
        zk.updateOrCreate(zkPath(), brokenIds.mkString(",").getBytes());
    }

    private ZkPath zkPath() {
        return calendarZkPath.child("broken-displays");
    }

    @Override
    public Schedule cronExpression() {
        return new SchedulePeriodic(Duration.standardSeconds(90));
    }

    @Override
    public Duration timeout() {
        return Duration.standardSeconds(90);
    }
}
