package ru.yandex.calendar.frontend.worker.task;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.calendar.frontend.worker.CalendarOnetimeTask;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.ics.feed.IcsFeedUpdater;
import ru.yandex.calendar.util.db.DbUtils;
import ru.yandex.commune.bazinga.pg.PgBazingaTaskManager;
import ru.yandex.commune.bazinga.pg.storage.PgBazingaStorage;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleLinear;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author dbrylev
 */
public class UpdateIcsFeedTask extends CalendarOnetimeTask<UpdateIcsFeedTask.Parameters> {

    public static final TaskQueueName QUEUE_NAME = new TaskQueueName("ics-feeds");

    @Autowired
    private IcsFeedUpdater icsFeedUpdater;

    @Autowired
    private PgBazingaStorage bazingaStorage;
    @Autowired
    private PgBazingaTaskManager bazingaTaskManager;

    @Autowired
    private TransactionTemplate transactionTemplate;

    public UpdateIcsFeedTask(long layerId) {
        super(new Parameters(layerId));
    }

    public UpdateIcsFeedTask() {
        super(Parameters.class);
    }

    @Override
    protected ActionSource getActionSource() {
        return ActionSource.WEB_ICS;
    }

    @Override
    protected void doExecute(Parameters parameters, ExecutionContext context) {
        icsFeedUpdater.updateIcsFeed(parameters.layerId).forEach(feed ->
                transactionTemplate.execute(DbUtils.transactionCallback(() -> {
                        bazingaStorage.deleteOnetimeJob(context.getFullJobId());
                        bazingaTaskManager.schedule(new UpdateIcsFeedTask(parameters.layerId), feed.getNextQueryTs());
                })));
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_FINISHED, ActiveUidDuplicateBehavior.MERGE_IF_READY);
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new RescheduleLinear(Duration.standardMinutes(10), Integer.MAX_VALUE);
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(20);
    }

    @Override
    public TaskQueueName queueName() {
        return QUEUE_NAME;
    }

    @Override
    public int priority() {
        return 0;
    }

    @BenderBindAllFields
    public static class Parameters {
        public final long layerId;

        public Parameters(long layerId) {
            this.layerId = layerId;
        }
    }
}
