package ru.yandex.calendar.frontend.worker.task;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.worker.CalendarOnetimeTask;
import ru.yandex.calendar.logic.XivaNotificationManager;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.ActionSource;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.CompoundReschedulePolicy;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleConstant;
import ru.yandex.commune.bazinga.scheduler.schedule.RescheduleExponential;
import ru.yandex.commune.bazinga.scheduler.schedule.ReschedulePolicy;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author dbrylev
 */
public class UpdateLayersTimestampsTask extends CalendarOnetimeTask<UpdateLayersTimestampsTask.Parameters> {
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private XivaNotificationManager xivaNotificationManager;


    public UpdateLayersTimestampsTask(long mainEventId, Option<String> connectionId, ActionInfo actionInfo) {
        super(new Parameters(mainEventId, connectionId, actionInfo.getNow(), actionInfo.getActionSource()));
    }

    public UpdateLayersTimestampsTask() {
        super(Parameters.class);
    }

    @Override
    public void doExecute(Parameters parameters, ExecutionContext context) {
        ListF<Long> layerIds = updateTimestamps(parameters);

        if (parameters.source != ActionSource.WEB_ICS && layerIds.isNotEmpty()) {
            xivaNotificationManager.notifyLayersUsersAboutEventsChangeAsynchronously(
                    layerIds, parameters.connectionId, ActionInfo.webTest(parameters.timestamp));
        }
    }

    public ListF<Long> updateTimestamps(Parameters parameters) {
        Instant ts = parameters.timestamp;
        ListF<Long> mainIds = Cf.list(parameters.mainEventId);

        return layerDao.updateStaleCollLastUpdateTsByMainEventIds(mainIds, ts)
                .plus(layerDao.updateStaleCollLastUpdateTsByMainEventIdsUsingEventLayersDeletedAt(mainIds, ts));
    }

    @Override
    public ReschedulePolicy reschedulePolicy() {
        return new CompoundReschedulePolicy(
                new RescheduleConstant(Duration.standardSeconds(15), 4),
                new RescheduleConstant(Duration.standardMinutes(1), 10),
                new RescheduleExponential(Duration.standardMinutes(1), 10));
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardSeconds(30);
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_RUNNING, ActiveUidDuplicateBehavior.DO_NOTHING);
    }

    @BenderBindAllFields
    public static class Parameters {
        public final long mainEventId;
        public final Option<String> connectionId;
        public final Instant timestamp;
        public final ActionSource source;

        public Parameters(long mainEventId, Option<String> connectionId, Instant timestamp, ActionSource source) {
            this.mainEventId = mainEventId;
            this.connectionId = connectionId;
            this.timestamp = timestamp;
            this.source = source;
        }
    }
}
