package ru.yandex.calendar.frontend.xiva;

import java.util.List;

import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.apache.http.impl.client.CloseableHttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.calendar.util.Environment;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClient4;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.io.http.client.entity.StringEntity;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @url http://wiki.yandex-team.ru/nordsturm/xiva-disk
 * @url http://wiki.yandex-team.ru/SergejjNikishin/xiva-schema
 */
public class Xiva {
    private static final Logger LOG = LoggerFactory.getLogger(Xiva.class);

    private static final String XIVA_REQUEST_TIMER = "application.request.time.xiva";
    private static final String XIVA_DISTRIBUTION_TIMER = "application.distribution.time.xiva.%s";
    private static final String USER_AGENT = "Yandex.Calendar";
    private final ApacheHttpClient4 httpClient =
            ApacheHttpClient4.wrap((CloseableHttpClient) (Environment.isProductionOrPre()
                    ? ApacheHttpClientUtils.multiThreadedHttpsClient(Timeout.seconds(5), 30, USER_AGENT)
                    : ApacheHttpClientUtils.trustAllMultiThreadedHttpsClient(Timeout.seconds(5), 30, 30, USER_AGENT)));
    @Autowired
    private XivaMetrics xivaMetrics;
    @Autowired
    private MeterRegistry registry;
    @Value("${xiva.push.api.url}")
    private String pushApiUrl;
    @Value("${xiva.push.api.token:-}")
    private String pushApiToken;

    public void push(PassportUid uid, String eventType, String message) {
        this.push(List.of(uid), eventType, message);
    }

    public void push(List<PassportUid> uids, String eventType, String message) {
        if (uids.isEmpty()) {
            return;
        }
        LOG.info("Starting xiva push send for {} users, {} eventType, {} message len", uids.size(), eventType, message.length());
        final long start = System.currentTimeMillis();
        registry.timer(String.format(XIVA_DISTRIBUTION_TIMER, eventType)).record(() -> {
            for (var uid : uids) {
                registry.timer(XIVA_REQUEST_TIMER).record(() -> send(uid, eventType, message));
            }
        });
        LOG.info("Xiva push send for {} users, {} eventType, {} message len finished in {} ms",
                uids.size(), eventType, message.length(), System.currentTimeMillis() - start);
    }

    private void send(PassportUid uid, String eventType, String message) {
        Check.isFalse(pushApiToken.isEmpty(), "Xiva is unavailable due to missing auth token");
        val url = UriBuilder.cons(pushApiUrl).appendPath("send")
                .addParam("stoken", pushApiToken)
                .addParam("uid", uid)
                .addParam("event", eventType).toUrl();
        try {
            httpClient.post(url, new StringEntity(message));
            xivaMetrics.onPush(HttpStatus.SC_200_OK);
        } catch (RuntimeException e) {
            xivaMetrics.onPush(e);
            throw e;
        }
    }

    @PreDestroy
    public void close() {
        httpClient.stop();
    }
}
