package ru.yandex.calendar.frontend.xiva;

import io.micrometer.core.instrument.MeterRegistry;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.commune.monitoring.Safe;
import ru.yandex.commune.monitoring.WithSensors;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.DistributionData;
import ru.yandex.misc.monica.core.blocks.Health;
import ru.yandex.misc.monica.core.blocks.MeterMapWithDistribution;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

@WithSensors
public class XivaMetrics implements MonicaContainer {
    private final static String XIVA_REQUEST_COUNTER = "application.request.xiva";
    private static final String HTTP_200_KEY = "200";
    @GroupByDefault
    @MonicaMetric(description = "Xiva push count (by http code)")
    private final MeterMapWithDistribution push = new MeterMapWithDistribution();
    @Autowired
    private MeterRegistry registry;

    @Safe
    @GroupByDefault
    @MonicaMetric(description = "Status of Xiva push for the last 15 minutes")
    public Health pushStatus() {
        return status(push);
    }

    private Health status(MeterMapWithDistribution mm) {
        double operationCount = mm.get(MetricName.EMPTY).apply().getAverage15Min() * 15 * 60;
        if (operationCount > 50) {
            DistributionData avg15minDist = mm.getAverage15minDistribution();
            double errorPercent = 100 - avg15minDist.fractions.getOrElse(HTTP_200_KEY, 100.0d);
            if (errorPercent > 25.0d) {
                return Health.unhealthy(formatMsg(operationCount, errorPercent));
            }
            if (errorPercent > 10.0d) {
                return Health.worried(formatMsg(operationCount, errorPercent));
            }
        }
        return Health.healthy();
    }

    private String formatMsg(double operationCount, double errorPercent) {
        return String.format(
                "%2.0f%% of errors in last 15 minutes, operations count=%2.0f%%", errorPercent, operationCount);
    }

    private void onPush(String key) {
        push.inc(key);
        push.inc(MetricName.EMPTY);
    }

    public void onPush(int httpStatusCode) {
        registry.counter(XIVA_REQUEST_COUNTER + "." + codeToKeyUnistat(httpStatusCode)).increment();
        onPush(codeToKey(httpStatusCode));
    }

    public void onPush(RuntimeException e) {
        if (e instanceof HttpException) {
            Option<Integer> statusCode = ((HttpException) e).getStatusCode();
            if (statusCode.isPresent()) {
                onPush(statusCode.get());
                return;
            }
        }
        registry.counter(XIVA_REQUEST_COUNTER + ".5xx").increment();
        onPush("error");
    }

    private String codeToKeyUnistat(int httpStatusCode) {
        return httpStatusCode / 100 + "xx";
    }

    private String codeToKey(int httpStatusCode) {
        return httpStatusCode == HttpStatus.SC_200_OK ? HTTP_200_KEY : httpStatusCode / 100 + "xx";
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName("xiva", new MetricName("xiva"), "Xiva statistics");
    }
}
