package ru.yandex.calendar.frontend.xiva.v2;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;

import com.google.common.base.Charsets;
import com.google.gson.Gson;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicNameValuePair;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.web.server.ResponseStatusException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.frontend.xiva.v2.models.XivaEntity;
import ru.yandex.calendar.frontend.xiva.v2.utils.HttpClientUtils;
import ru.yandex.calendar.frontend.xiva.v2.utils.UriUtils;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

public class XivaHttpClient {
    private static final Logger logger = LoggerFactory.getLogger(XivaHttpClient.class);

    private final String service = "yandex-team-app";
    @Value("${xiva.v2.push.api.url}")
    private String baseUri;
    @Value("${xiva.app.push.api.token:-}")
    private String sendToken;

    private final Duration connectTimeout = new Duration(15000L);
    private final Duration socketTimeout = new Duration(15000L);
    private static final String USER_AGENT = "Yandex.Calendar";

    private final Gson gson = new Gson();
    private final HttpClient httpClient = HttpClientUtils.https(new Timeout(socketTimeout.getMillis(),
            connectTimeout.getMillis()), 30, USER_AGENT);

    public void postNotification(long uid, String eventName, XivaEntity message, int ttl) {
        UriBuilder uriBuilder = UriBuilder.cons(baseUri);
        uriBuilder.appendPath("/send");
        uriBuilder.addParam("uid", uid);
        uriBuilder.addParam("event", eventName);
        uriBuilder.addParam("token", sendToken);
        uriBuilder.addParam("ttl", ttl);

        HttpPost request = new HttpPost(uriBuilder.build());
        request.setConfig(HttpClientUtils.requestConfig(connectTimeout, socketTimeout));

        try {
            byte[] bytes = gson.toJson(message).getBytes(StandardCharsets.UTF_8);
            request.setEntity(new ByteArrayEntity(bytes, ContentType.APPLICATION_JSON));
            logger.info("Request: {}, Entity: {}",
                    UriUtils.removeToken(request.getRequestLine().toString(), Option.of(sendToken)),
                    message);

            executeRequestOrThrow(request);
        } catch (Exception e) {
            logger.error("Error in xiva client, event: " + eventName, e);
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    public void subscribeApp(String oauthToken,
                             Long uid,
                             String appName,
                             String platform,
                             String uuid,
                             String pushToken,
                             Option<String> filter,
                             Option<String> extra) {
        UriBuilder uriBuilder = UriBuilder.cons(baseUri);
        uriBuilder.appendPath("/subscribe/app");
        uriBuilder.addParam("service", String.format("%s", service));
        uriBuilder.addParam("user", uid);
        uriBuilder.addParam("client", "yandex_team_app_calendar");
        uriBuilder.addParam("uuid", uuid);
        uriBuilder.addParam("app_name", appName);
        uriBuilder.addParam("platform", platform);
        URI uri = uriBuilder.build();

        HttpPost request = new HttpPost(uri);
        request.setConfig(HttpClientUtils.requestConfig(connectTimeout, socketTimeout));

        ListF<BasicNameValuePair> params = Cf.arrayList(new BasicNameValuePair("push_token", pushToken));
        filter.forEach(v -> params.add(new BasicNameValuePair("filter", v)));
        extra.forEach(v -> params.add(new BasicNameValuePair("extra", v)));
        request.setEntity(new UrlEncodedFormEntity(params, Charsets.UTF_8));

        request.addHeader("Authorization", "OAuth " + oauthToken);

        try {
            logger.info("Request: {}", UriUtils.removeToken(request.getRequestLine().toString(), Option.of(oauthToken)));
            executeRequestOrThrow(request);
        } catch (Exception e) {
            logger.error("Error in xiva client (subscribe app)", e);
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    private void executeRequestOrThrow(HttpPost request) {
        try {
            httpClient.execute(request, response -> {
                HttpStatus statusCode = HttpStatus.resolve(response.getStatusLine().getStatusCode());
                if (statusCode == null) {
                    statusCode = HttpStatus.INTERNAL_SERVER_ERROR;
                }

                logger.info("Xiva command: {} response: {}", request.getRequestLine(), statusCode);

                if (statusCode != HttpStatus.OK) {
                    logger.error("Error on xiva command. Xiva responded with code={} reasonPhrase={} entity={}",
                            statusCode, response.getStatusLine().getReasonPhrase(),
                            response.getEntity());
                    throw new ResponseStatusException(statusCode);
                }
                return response;
            }, new HttpClientContext());
        } catch (IOException e) {
            logger.error("Error in xiva client, command " + request.getRequestLine(), e);
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }
}
