package ru.yandex.calendar.frontend.xiva.v2.utils;

import org.apache.http.client.config.CookieSpecs;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.joda.time.Duration;

import ru.yandex.misc.io.http.Timeout;

public class HttpClientUtils {
    public static CloseableHttpClient https(Timeout timeout, int maxConnectionCount, String userAgent) {
        return new Builder()
                .timeout(timeout)
                .maxConnectionCount(maxConnectionCount)
                .userAgent(userAgent)
                .cookieSpec(CookieSpecs.BROWSER_COMPATIBILITY)
                .build();
    }

    public static RequestConfig requestConfig(Duration connectTimeout, Duration socketTimeout) {
        int connectTimeoutMillis = intMillis(connectTimeout.getMillis());
        int soTimeoutMillis = intMillis(socketTimeout.getMillis());
        return RequestConfig.custom()
                .setStaleConnectionCheckEnabled(true)
                .setConnectionRequestTimeout(connectTimeoutMillis)
                .setConnectTimeout(connectTimeoutMillis)
                .setSocketTimeout(soTimeoutMillis)
                .setCookieSpec(CookieSpecs.BROWSER_COMPATIBILITY)
                .build();
    }

    public static int intMillis(long millis) {
        return (int) (Math.min(millis, Integer.MAX_VALUE));
    }


    private static final class Builder {
        private final PoolingHttpClientConnectionManager connManager;
        private final RequestConfig.Builder requestConfigBuilder;
        private final SocketConfig.Builder defaultSocketConfigBuilder;
        private final HttpClientBuilder httpClientBuilder;

        Builder() {
            Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create()
                    .register("http", PlainConnectionSocketFactory.getSocketFactory())
                    .register("https", SSLConnectionSocketFactory.getSocketFactory())
                    .build();
            connManager = new PoolingHttpClientConnectionManager(registry);
            requestConfigBuilder = RequestConfig.custom()
                    .setStaleConnectionCheckEnabled(true);
            defaultSocketConfigBuilder = SocketConfig.custom();
            httpClientBuilder = HttpClientBuilder.create();
        }

        public Builder timeout(Timeout timeout) {
            int connectTimeoutMillis = intMillis(timeout.getConnectTimeoutMilliseconds());
            int soTimeoutMillis = intMillis(timeout.getSoTimeoutMilliseconds());
            requestConfigBuilder
                    .setConnectionRequestTimeout(connectTimeoutMillis)
                    .setConnectTimeout(connectTimeoutMillis)
                    .setSocketTimeout(soTimeoutMillis);
            defaultSocketConfigBuilder
                    .setSoTimeout(soTimeoutMillis);
            return this;
        }

        public Builder maxConnectionCount(int maxConnectionCount) {
            connManager.setDefaultMaxPerRoute(maxConnectionCount);
            connManager.setMaxTotal(maxConnectionCount);
            return this;
        }

        public Builder userAgent(String userAgent) {
            httpClientBuilder.setUserAgent(userAgent);
            return this;
        }

        public Builder cookieSpec(String cookieSpec) {
            requestConfigBuilder.setCookieSpec(cookieSpec);
            return this;
        }

        public CloseableHttpClient build() {
            RequestConfig requestConfig = requestConfigBuilder.build();
            SocketConfig socketConfig = defaultSocketConfigBuilder.build();
            connManager.setDefaultSocketConfig(socketConfig);

            return httpClientBuilder
                    .setDefaultRequestConfig(requestConfig)
                    .setConnectionManager(connManager)
                    .build();
        }
    }
}
