package ru.yandex.calendar.frontend.yamb;

import java.io.IOException;
import java.io.InputStream;

import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.SneakyThrows;
import lombok.val;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.commune.json.JsonNumber;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.Check;

/**
 * @url https://github.yandex-team.ru/pages/yamb/yamb-bots/api/bot/methods/
 */
public class Yamb {
    private final HttpClient httpClient;
    @Value("${yamb.url}")
    private String url;
    @Value("${yamb.token:-}")
    private String token;
    @Value("${yamb.organization}")
    private int organization;

    public Yamb(HttpClientConfiguration conf, MeterRegistry registry) {
        this.httpClient = conf.authenticatingClient(new YambAuthenticationHandler(() -> token), registry, "yamb");
    }

    @SneakyThrows
    public void sendMessage(PassportUid uid, String message) {
        Check.isFalse(token.isEmpty(), "Yamb is unavailable due to missing auth token");

        val url = UriBuilder.cons(this.url).appendPath("/bot/send-message/").toUrl();

        val request = new HttpPost(url);

        request.setEntity(new StringEntity(new JsonObject(Tuple2List.fromPairs(
                "user_id", JsonNumber.valueOf(uid.getUid()),
                "organization_id", JsonNumber.valueOf(organization),
                "text", JsonString.valueOf(message))).serialize(), ContentType.APPLICATION_JSON));

        httpClient.execute(request, new StringResponseHandler());
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }

    public static class StringResponseHandler implements ResponseHandler<String> {
        public String handleResponse(HttpResponse response) throws IOException {
            val statusCode = response.getStatusLine().getStatusCode();
            if (!ru.yandex.misc.io.http.HttpStatus.is2xx(statusCode)) {
                throw new StrangeResponseException(statusCode, response.getEntity().getContent());
            }
            return InputStreamSourceUtils.wrap(response.getEntity().getContent())
                    .readText();
        }
    }

    public static class StrangeResponseException extends HttpException {
        public StrangeResponseException(int statusCode, InputStream response) {
            super(statusCode, InputStreamSourceUtils.wrap(response).readText());
        }
    }
}
