package ru.yandex.calendar.frontend.ywmi;

import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.MeterRegistry;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.util.EntityUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.BenderParserUtils;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.CharsetUtils;

/**
 * @url https://wiki.yandex-team.ru/users/jkennedy/ywmiapi/
 */
public class Ywmi {
    private static final BenderMapper mapper = new BenderMapper();
    private final String url;
    private final HttpClient httpClient;

    public Ywmi(String url, HttpClientConfiguration httpConf, MeterRegistry registry) {
        this.url = url;
        this.httpClient = httpConf.consClient(registry, "ywmi");
    }

    public Envelope filterSearch(PassportUid uid, long mid) {
        Envelopes envelopes = filterSearch(uid, Cf.list(mid));

        if (envelopes.envelopes.isEmpty()) {
            throw new EnvelopeNotFoundException("Envelope not found for " + uid + " by mid " + mid);
        }
        return envelopes.envelopes.single();
    }

    public Envelopes filterSearch(PassportUid uid, ListF<Long> mids) {
        if (mids.isEmpty()) return new Envelopes(Cf.list());

        HttpGet request = new HttpGet(UriBuilder.cons(url + "/filter_search")
                .addParam("uid", uid.getUid())
                .addParams(mids.toTuple2List(m -> "mids", m -> Long.toString(m)))
                .addParams(Option.when(uid.isYandexTeamRu(), "default").toTuple2List(fs -> "folder_set", fs -> fs))
                .build());

        return ApacheHttpClientUtils.execute(request, httpClient, new ParseResponseHandler<>(Envelopes.class));
    }

    public Mids messagesByMessageId(PassportUid uid, String messageId) {
        HttpGet request = new HttpGet(UriBuilder.cons(url + "/messages_by_message_id")
                .addParam("uid", uid.getUid())
                .addParam("msgid", messageId).build());

        return ApacheHttpClientUtils.execute(request, httpClient, new ParseResponseHandler<>(Mids.class));
    }

    public static class ParseResponseHandler<T> implements ResponseHandler<T> {
        private final Class<T> clazz;

        public ParseResponseHandler(Class<T> clazz) {
            this.clazz = clazz;
        }

        @Override
        public T handleResponse(HttpResponse response) {
            Option<String> text = Option.empty();

            try {
                text = Option.of(EntityUtils.toString(response.getEntity(), CharsetUtils.UTF8_CHARSET));
                BenderJsonNode json = BenderParserUtils.json(text.get());

                if (response.getStatusLine().getStatusCode() == 200 && !json.getField("error").isPresent()) {
                    return mapper.parseJson(clazz, json);
                }
            } catch (Exception e) {
                throw new StrangeResponseException(response.getStatusLine().getStatusCode(), text.getOrNull(), e);
            }
            throw new StrangeResponseException(response.getStatusLine().getStatusCode(), text.getOrNull());
        }
    }

    public static class EnvelopeNotFoundException extends RuntimeException {
        public EnvelopeNotFoundException(String message) {
            super(message);
        }
    }

    public static class StrangeResponseException extends HttpException {
        public StrangeResponseException(int statusCode, String response) {
            super(statusCode, response);
        }

        public StrangeResponseException(int statusCode, String response, Throwable cause) {
            super(statusCode, response, cause);
        }
    }

    @PreDestroy
    public void close() {
        ApacheHttpClientUtils.stopQuietly(httpClient);
    }
}
