package ru.yandex.calendar.logic;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.worker.task.UpdateLayersTimestampsTask;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.event.dao.EventDao;
import ru.yandex.calendar.logic.event.dao.MainEventDao;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.env.EnvironmentType;

/**
 * @author gutman
 */
@Slf4j
public class LastUpdateManager {

    @Autowired
    private LayerDao layerDao;
    @Autowired
    private MainEventDao mainEventDao;
    @Autowired
    private EventDao eventDao;
    @Autowired
    private EnvironmentType environmentType;
    @Autowired
    private BazingaTaskManager bazingaTaskManager;
    @Autowired
    private UpdateLayersTimestampsTask updateTask;
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;


    public void updateTimestampsAsync(ListF<Long> mainEventIds, ActionInfo actionInfo) {
        mainEventIds.forEach(mainEventId -> updateTimestampsAsync(mainEventId, actionInfo));
    }

    public void updateTimestampsAsync(long mainEventId, ActionInfo actionInfo) {
        UpdateLayersTimestampsTask task = new UpdateLayersTimestampsTask(
                mainEventId, CalendarRequest.getCurrentConnectionId(), actionInfo);

        if (environmentType == EnvironmentType.TESTS) {
            mainEventDao.updateStaleLastUpdateTsByIds(Cf.list(mainEventId), actionInfo.getNow());
            updateTask.updateTimestamps(task.getParametersTyped());

        } else if (passportAuthDomainsHolder.containsYandexTeamRu()) {
            mainEventDao.updateStaleLastUpdateTsByIds(Cf.list(mainEventId), actionInfo.getNow());
            log.debug("updateTimestampsAsync for: " + mainEventId
                    + ", action: " + actionInfo.getAction()
                    + ", source: " + actionInfo.getActionSource()
                    + ", request: " + actionInfo.getRequestIdWithHostId());
            bazingaTaskManager.schedule(task);

        } else {
            updateMainEventAndLayerTimestampsByMainEventIds(Cf.list(mainEventId), actionInfo);
        }
    }

    public void updateMainEventAndLayerTimestamps(ListF<Long> eventIds, ActionInfo actionInfo) {
        updateMainEventAndLayerTimestampsByMainEventIds(eventDao.findMainEventIdsByEventIds(eventIds), actionInfo);
    }

    public void updateMainEventAndLayerTimestampsByMainEventIds(ListF<Long> mainEventIds, ActionInfo actionInfo) {
        ListF<Long> mastersAndRecurrencesIds = eventDao.findEventIdsByMainEventIds(mainEventIds);
        mainEventDao.updateMainEventLastUpdateTsByMainEventIds(mainEventIds, actionInfo.getNow());
        layerDao.updateLayersCollLastUpdateTsByEventIds(actionInfo.getNow(), mastersAndRecurrencesIds);
    }

    public void updateMainEventAndLayerTimestampsByLayerId(long layerId, ActionInfo actionInfo) {
        mainEventDao.updateMainEventLastUpdateTsByLayerId(layerId, actionInfo.getNow());
        layerDao.updateLayerCollLastUpdateTsByLayerId(actionInfo.getNow(), layerId);
    }

    public void updateLayerTimestamps(ListF<Long> layerIds, ActionInfo actionInfo) {
        layerDao.updateLayerCollLastUpdateTsByLayerIds(actionInfo.getNow(), layerIds);
    }

}
