package ru.yandex.calendar.logic;

import java.util.List;
import java.util.function.Consumer;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.CalendarRequest;
import ru.yandex.calendar.frontend.xiva.Xiva;
import ru.yandex.calendar.logic.event.ActionInfo;
import ru.yandex.calendar.logic.layer.LayerDao;
import ru.yandex.calendar.logic.notification.xiva.XivaSendNotificationExecutor;
import ru.yandex.calendar.util.exception.ExceptionUtils;
import ru.yandex.commune.json.JsonObject;
import ru.yandex.commune.json.JsonString;
import ru.yandex.commune.json.JsonValue;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.env.EnvironmentType;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.log.mlf.ndc.Ndc;
import ru.yandex.misc.log.reqid.RequestIdStack;
import ru.yandex.misc.thread.WhatThreadDoes;

/**
 * @author dbrylev
 */
public class XivaNotificationManager {
    private static final Logger logger = LoggerFactory.getLogger(XivaNotificationManager.class);

    @Autowired
    private XivaSendNotificationExecutor executor;
    @Autowired
    private LayerDao layerDao;
    @Autowired
    private Xiva xiva;
    @Autowired
    private EnvironmentType environmentType;

    private void notifyLayersUsersAboutEventsChange(ListF<PassportUid> uids, Option<String> connectionId) {
        if (environmentType == EnvironmentType.TESTS) return;
        try {
            JsonObject message = new JsonObject(Tuple2List.<String, JsonValue>tuple2List()
                    .plus(connectionId.map(id -> Tuple2.tuple("connectionId", JsonString.valueOf(id)))));

            xiva.push(uids, "user-events-changed", message.serialize());
        } catch (Exception e) {
            ExceptionUtils.rethrowIfTlt(e);
            logger.error("Failed to send xiva notification to users {}: {}", uids, e);
        }
    }

    public void notifyLayersUsersAboutEventsChange(ListF<Long> layerIds, ActionInfo actionInfo) {
        logger.info("Sending xiva push notification synchronously! caused by action {}/{}, stack {}",
                actionInfo.getAction(), actionInfo.getActionSource(), List.of(Thread.currentThread().getStackTrace()));
        ListF<PassportUid> uids = layerDao.findLayerUserUidsByLayerIdsModifiedSince(layerIds, actionInfo.getNow());
        notifyLayersUsersAboutEventsChange(uids, CalendarRequest.getCurrentConnectionId());
    }

    public void notifyLayersUsersAboutEventsChangeAsynchronously(ListF<Long> layerIds, ActionInfo actionInfo) {
        notifyLayersUsersAboutEventsChangeAsynchronously(layerIds, CalendarRequest.getCurrentConnectionId(), actionInfo);
    }

    public void notifyLayersUsersAboutEventsChangeAsynchronously(
            ListF<Long> layerIds, Option<String> connectionId, ActionInfo actionInfo)
    {
        if (environmentType == EnvironmentType.TESTS) return;
        if (layerIds.isEmpty()) return;

        Option<String> ndc = Option.ofNullable(Ndc.get());
        ListF<PassportUid> uidsToSend = layerDao.findLayerUserUidsByLayerIdsModifiedSince(layerIds, actionInfo.getNow());

        logger.info("Sending xiva push notification asynchronously caused by action {}/{}", actionInfo.getAction(), actionInfo.getActionSource());
        Consumer<ListF<PassportUid>> send = (uidsBatch) ->
                send(uidsBatch, ndc, connectionId, actionInfo);

        executor.schedule(uidsToSend, send);
    }

    private void send(ListF<PassportUid> uids, Option<String> ndc, Option<String> connectionId, ActionInfo actionInfo) {
        Option<Ndc.Handle> ndcHandle = ndc.map(Ndc::pushReplace);
        WhatThreadDoes.Handle wtdHandle = WhatThreadDoes.push("sends xiva notifications");
        RequestIdStack.Handle ridHandle = RequestIdStack.pushReplace(actionInfo.getRequestId());

        try {
            notifyLayersUsersAboutEventsChange(uids, connectionId);
        } catch (Exception e) {
            logger.error("Error occurred while sending notifications", e);
        } finally {
            wtdHandle.popSafely();
            ridHandle.popSafely();
            ndcHandle.forEach(Ndc.Handle::popSafely);
        }
    }
}
