package ru.yandex.calendar.logic.beans;

import java.util.Map;

import org.jdom.Element;
import org.joda.time.DateTimeZone;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.calendar.util.xml.CalendarXmlizer;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.mapObject.MapObject;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * Base class for all beans. A bean is a light wrapper to a database instance
 * with convenient getters, setters, insert/update/delete routines
 * and different checks (field existence, nullability, obligation etc.)
 *
 * @author ssytnik
 */
public abstract class Bean<TId> extends MapObject {
    private static final Logger logger = LoggerFactory.getLogger(Bean.class);

    //// Initialization

    private boolean lockedForUpdate;

    protected Bean() {
    }

    @Override
    public Bean clone() {
        return (Bean) super.clone();
    }

    public <A extends Bean> A withoutId() {
        A res = super.<A>copy();
        res.unsetField(getMapObjectDescription().getFieldByName("id"));
        res.lock();
        return res;
    }

    public <A extends Bean> A withFields(A source) {
        A copy = copy();
        copy.setFields(source);
        return copy;
    }

    // Getters

    public abstract BeanHelper<? extends Bean<TId>, TId> getHelper();

    // used?
    public Map<String, Object> getValues() {
        MapF<String, Object> r = Cf.hashMap();
        for (Tuple2<MapField<?>, Object> v : this.getFieldValues()) {
            r.put(v.get1().column().name(), v.get2());
        }
        return r;
    }

    public abstract TId getId();

    public abstract void setId(TId id);

    public Object get(String field) {
        return getFieldValue(getMapObjectDescription().getFieldByName(field));
    }

    public <T> Option<T> getValueIfSet(MapField<T> field) {
        return isFieldSet(field) ? getFieldValueNullAsNone(field) : Option.empty();
    }

    public void validateIsLockedForUpdate() {
        Validate.isTrue(lockedForUpdate, "Event not locked for update: ", getId());
    }

    @SuppressWarnings("unchecked")
    public void set(String field, Object value) {
        MapField<Object> mapField = (MapField<Object>) getMapObjectDescription().getFieldByName(field);
        setFieldValue(mapField, value);
    }

    //// Update

    public void appendXmlTo(Element element, DateTimeZone tz, MapField<?>... fields) {
        for (MapField<?> field : fields) {
            String elementName = CamelWords.parse(field.getName()).toXmlName();
            CalendarXmlizer.appendDtfElm(element, elementName, this.getFieldValue(field), tz);
        }
    }

    // rootElement +new_this_element, returns new_this_element
    public Element addXmlElementTo(Element rootElement, DateTimeZone tz) {
        Element res = toXmlElement(tz);
        rootElement.addContent(res);
        return res;
    }

    // rootElement +new_this_element(itemTagName), returns new_this_element
    public void addXmlElementTo(Element rootElement, String itemTagName, DateTimeZone tz)
            {
        Element res = toXmlElement(itemTagName, tz);
        rootElement.addContent(res);
    }

    public Element toXmlElement(DateTimeZone tz) {
        return toXmlElement(getHelper().getItemTagName(), tz);
    }

    public Element toXmlElement(String itemTagName, DateTimeZone tz) {
        Element element = new Element(itemTagName);

        appendXmlTo(element, tz, getFieldValues().get1().toArray(new MapField<?>[0]));

        return element;
    }

    /**
     * Safely checks if field is going to change, provided with old and new beans.
     * NOTE: at the moment, both beans should be set (not null).
     */
    public static <T extends Bean> boolean isFieldChanging(T newBean, T oldBean, MapField<?> field) {
        if (!newBean.isFieldSet(field)) { return false; }
        if (!oldBean.isFieldSet(field)) {
            throw new IllegalStateException(
                    "field " + field + " must be set in the old bean to perform comparison");
        }
        return !ObjectUtils.equals((Object) oldBean.getFieldValue(field), (Object) newBean.getFieldValue(field));
    }

    public void setLockedForUpdate() {
        lockedForUpdate = true;
    }
}
