package ru.yandex.calendar.logic.beans;

import java.sql.ResultSet;
import java.sql.SQLException;

import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.jdbc.core.RowMapper;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.calendar.logic.beans.generated.Settings;
import ru.yandex.calendar.util.base.Cf2;
import ru.yandex.calendar.util.db.BeanRowMapper;
import ru.yandex.calendar.util.db.ColumnsInSetCondition;
import ru.yandex.commune.mapObject.MapField;
import ru.yandex.commune.mapObject.MapObject;
import ru.yandex.commune.mapObject.MapObjectDescription;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.InSetCondition;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.resultSet.RowMapperSupport;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.reflection.ClassX;

/**
 * A bean helper base class. Acts like a single entity shared by multiple bean instances.
 * Each inheritor only helps to a specific bean type specified by template parameter T.
 *
 * This class contains information about all fields, nullable fields, optional fields,
 * and sql type groups for the entity type being served, as well as performs auxiliary
 * actions like preparing SQL queries or checking fields accuracy.
 *
 * @author ssytnik
 */
public abstract class BeanHelper<T extends Bean<TId>, TId> {
    public String getTableName() {
        return beanMapObjectDescription().getTableName();
    }

    // FACTORIES for beans...

    private final Function0<T> constructor = Cf2.consF(beanClass());

    public T createBean() {
        return constructor.apply();
    }

    protected ClassX<T> beanClass() {
        return beanMapObjectDescription().getClazz().get().uncheckedCast();
    }

    public abstract MapObjectDescription beanMapObjectDescription();

    protected static Function<BeanHelper<?, ?>, ClassX<Bean>> beanClassXF() {
        return bh -> bh.beanClass().uncheckedCast();
    }

    public RowMapperSupport<T> beanRowMapper() {
        return beanMapObjectDescription().rowMapper();
    }

    public RowMapperSupport<T> beanRowMapper(String prefix) {
        return beanMapObjectDescription().rowMapper(prefix);
    }

    public BeanRowMapper<T> offsetRowMapper(int columnOffset) {
        return new BeanRowMapper<T>(this, columnOffset);
    }

    public BeanRowMapper<T> offsetRowMapper(int columnOffset, ListF<MapField<?>> fields) {
        return new BeanRowMapper<T>(this, columnOffset, fields);
    }

    public SqlCondition idEqCondition(TId id) {
        return idInSetCondition(Cf.list(id));
    }

    public SqlCondition idInSetCondition(CollectionF<TId> ids) {
        return fieldsInSetCondition(beanMapObjectDescription().getIdFields(), ids);
    }

    public static SqlCondition fieldsInSetCondition(ListF<MapField<?>> fields, CollectionF<?> values) {
        ListF<?> unique = values.stableUnique().toList();
        ListF<ConditionUtils.Column> columns = fields.map(MapField::column);

        if (unique.isEmpty() || columns.isEmpty()) {
            return SqlCondition.falseCondition();

        } else if (columns.size() == 1) {
            return values.size() == 1
                    ? columns.single().eq(values.single())
                    : new InSetCondition(columns.single(), values, true);

        } else {
            return new ColumnsInSetCondition(columns, unique);
        }
    }

    public String getItemTagName() {
        return CamelWords.parse(getTableName()).toXmlName();
    }

    private static final ListF<BeanHelper<?, ?>> beanHelpers = findBeanHelpers();

    private static ListF<BeanHelper<?, ?>> findBeanHelpers() {
        try {
            ListF<BeanHelper<?, ?>> r = Cf.arrayList();
            PathMatchingResourcePatternResolver resolve = new PathMatchingResourcePatternResolver();
            Resource[] resources = resolve.getResources("classpath:ru/yandex/calendar/logic/beans/generated/*Helper.class");
            for (Resource resource : resources) {
                String name = resource.getURI().toString()
                    .replaceFirst(".*/generated/", "")
                    .replaceFirst("\\.class$", "");
                ClassX<?> clazz = ClassX.forName(Settings.class.getPackage().getName() + "." + name);
                r.add((BeanHelper<?, ?>) clazz.newInstance());
            }
            return r;
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private static final MapF<ClassX<Bean>, BeanHelper<?, ?>> beanHelperByBeanClassName =
        beanHelpers.zipWith(beanClassXF()).invert().toMap();

    public static BeanHelper<?, ?> forBean(ClassX<? extends Bean> beanType) {
        return beanHelperByBeanClassName.getOrThrow(beanType.<Bean>uncheckedCast());
    }

    public T findChanges(T oldBean, T newBean) {
        T bean = createBean();
        for (MapField<?> field : bean.getMapObjectDescription().getFields()) {
            if (Bean.isFieldChanging(newBean, oldBean, field)) {
                bean.setFieldFromIfDefined(field, newBean);
            }
        }
        bean.lock();
        return bean;
    }

    public static Function<MapField<?>, String> columnNameF(String prefix) {
        return f -> prefix + f.getDatabaser().columnNames().single();
    }

    public static String columns(ListF<MapField<?>> fields, String prefix) {
        return fields.map(columnNameF(prefix)).mkString(", ");
    }

    public static <T extends Bean> T populateFields(
            Class<T> objectClass, ListF<MapField<?>> fields, ResultSet rs, String prefix)
    {
        T object = (T) forBean(ClassX.wrap(objectClass)).createBean();
        fields.forEach(f -> populateField(object, f, rs, prefix));
        return object;
    }

    public static <T extends Bean> RowMapper<T> fieldsRowMapper(
            Class<T> beanType, ListF<MapField<?>> fields, String prefix)
    {
        return (rs, num) -> populateFields(beanType, fields, rs, prefix);
    }

    public static ListF<MapField<?>> getFields(Class<? extends Bean> beanType) {
        return forBean(ClassX.wrap(beanType)).beanMapObjectDescription().getFields();
    }

    @SuppressWarnings("unchecked")
    private static void populateField(MapObject o, MapField<?> field, ResultSet rs, String prefix) {
        try {
            field.getDatabaser().populateFromResultSet(o, (MapField) field, rs, prefix);
        } catch (SQLException e) {
            throw ru.yandex.calendar.util.exception.ExceptionUtils.throwException(e);
        }
    }

} //~
