package ru.yandex.calendar.logic.contact;

import java.util.Optional;
import java.util.regex.Pattern;

import io.micrometer.core.instrument.MeterRegistry;
import lombok.val;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.calendar.logic.contact.addressbook.AddressBook;
import ru.yandex.calendar.logic.contact.directory.DirectoryManager;
import ru.yandex.calendar.logic.contact.ml.Ml;
import ru.yandex.calendar.tvm.TvmClient;
import ru.yandex.calendar.util.HttpClientConfiguration;
import ru.yandex.calendar.util.HttpClientWithMetrics;
import ru.yandex.misc.io.http.Timeout;

import static ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils.multiThreadedHttpsClient;

@Configuration
public class ContactContextConfiguration {
    private static final Pattern COLLIE_METRICS_URI = Pattern.compile(
        ".*(/carddav/propfind|/carddav/multiget|/compat/colabook_feed_addrdb)|.*(/carddav).+(/put|/delete)|.+(/location)$"
    );

    @Bean
    public ContactRoutines contactRoutines() {
        return new ContactRoutines();
    }

    private static HttpClientWithMetrics createClientFor(String service, int maxConnections, Timeout timeout,
                                                         MeterRegistry registry, Optional<Pattern> metricsUriPattern) {
        val client = multiThreadedHttpsClient(timeout, maxConnections, "Yandex.Calendar");
        return new HttpClientWithMetrics(client, registry, service, metricsUriPattern);
    }

    @Bean
    public AddressBook addressBook(
            @Value("${address_book.collie-url}") String collieUrl,
            @Value("${address_book.collie-tvm-id}") int collieTvmId,
            @Value("${address_book.search-url}") String searchUrl,
            @Value("${address_book.max_connections}") int maxConnections,
            @Value("${address_book.connection_timeout}") long connTimeout,
            @Value("${address_book.socket_timeout}") long soTimeout,
            MeterRegistry registry, TvmClient tvmClient) {
        val timeout = new Timeout(soTimeout, connTimeout);
        val collieHttpClient = createClientFor("collie", maxConnections, timeout, registry, Optional.of(COLLIE_METRICS_URI));
        val searchHttpClient = createClientFor("contactsSearch", maxConnections, timeout, registry, Optional.empty());

        return new AddressBook(collieUrl, collieTvmId, searchUrl, collieHttpClient,
            searchHttpClient, tvmClient);
    }

    @Bean
    public DirectoryManager directoryManager() {
        return new DirectoryManager();
    }

    @Bean
    public Ml ml(
            @Value("${yt.ml.url}") String baseUrl,
            @Value("${yt.ml.tvm-id}") int mlTvmId,
            MeterRegistry registry,
            TvmClient tvmClient)
    {
        return new Ml(baseUrl, HttpClientConfiguration.fromProperties("yt.ml"), registry, tvmClient, mlTvmId);
    }
}
