package ru.yandex.calendar.logic.contact;

import java.util.List;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;
import lombok.val;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectorsF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.calendar.logic.contact.addressbook.AddressBook;
import ru.yandex.calendar.logic.domain.PassportAuthDomainsHolder;
import ru.yandex.calendar.logic.resource.ResourceRoutines;
import ru.yandex.calendar.logic.user.Language;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.env.EnvironmentType;

/**
 * Manages user contacts and invitations.
 *
 * Responsible for retrieving information about contacts for some yandex user
 * (using external sources as well as inner, added by calendar itself,
 * contacts), sending it to the makeup and allowing makeup to invite some of the
 * contacts. Also contains methods for adding (by calendar, after invitation is
 * accepted) and removing (by the user, manually) custom contacts (emails).
 *
 * We were going to store (cache) external contacts in the calendar, but it seems
 * that external source does that good enough. Therefore we can rely on it.
 */
@Slf4j
public class ContactRoutines {
    @Autowired
    private PassportAuthDomainsHolder passportAuthDomainsHolder;
    @Autowired
    private UserManager userManager;
    @Autowired
    private ResourceRoutines resourceRoutines;
    @Autowired
    private AddressBook addressBook;
    @Autowired
    private EnvironmentType environmentType;

    public ListF<Contact> findUserContactsByEmails(PassportUid uid, List<Email> emails) {
        if (environmentType == EnvironmentType.TESTS) return Cf.list();
        try {
            return addressBook.findUserContactsByEmails(uid, emails);
        } catch (Exception e) {
            log.error("Failed to find user contacts: {}", ExceptionUtils.getAllMessages(e));
            return Cf.list();
        }
    }

    public void exportUserContactsEmails(PassportUid uid, ListF<Email> contacts) {
        if (environmentType == EnvironmentType.TESTS) {
            return;
        }

        val rejected = StreamEx.of(resourceRoutines.selectResourceEmails(contacts))
            .append(userManager.getEmailByUid(uid).stream())
            .toImmutableList();
        contacts = contacts.unique().minus(rejected).toList();

        if (contacts.isNotEmpty()) {
            try {
                addressBook.exportUserContactsEmails(uid, contacts);
            } catch (Exception e) {
                log.warn("Failed to export user contacts: {}", ExceptionUtils.getAllMessages(e));
            }
        }
    }

    public Tuple2List<Email, Option<NameI18n>> getI18NamesByEmails(PassportUid uid, List<Email> emails) {
        Tuple2List<Email, NameI18n> found = Tuple2List.tuple2List();
        var notFound = emails;

        if (passportAuthDomainsHolder.containsYandexTeamRu()) {
            val nameByEmail = StreamEx.of(notFound)
                .distinct()
                .toMap(userManager::getYtUserNameByEmail);

            found = EntryStream.of(nameByEmail)
                .flatMapValues(Optional::stream)
                .mapKeyValue(Tuple2::tuple)
                .collect(CollectorsF.toTuple2List());

            notFound = EntryStream.of(nameByEmail)
                .filterValues(Optional::isEmpty)
                .keys()
                .toImmutableList();
        }

        Tuple2List<Email, NameI18n> x = findUserContactsByEmails(uid, notFound)
                .toTuple2List(Contact::getEmail, contact -> NameI18n.constant(contact.getName()));

        return StreamEx.of(emails)
            .mapToEntry(found.plus(x).toMap()::getO)
            .mapKeyValue(Tuple2::tuple)
            .collect(CollectorsF.toTuple2List());
    }

    public Tuple2List<Email, Option<String>> getNamesByEmails(PassportUid uid, ListF<Email> emails) {
        return getI18NamesByEmails(uid, emails).map2(a -> a.map(NameI18n.getNameF(Language.RUSSIAN)));
    }
}
