package ru.yandex.calendar.logic.contact;

import lombok.Getter;

import ru.yandex.bolts.collection.Option;
import ru.yandex.calendar.logic.sending.param.Sender;
import ru.yandex.calendar.logic.sharing.participant.ParticipantInfo;
import ru.yandex.calendar.logic.user.NameI18n;
import ru.yandex.calendar.logic.user.UserManager;
import ru.yandex.calendar.micro.yt.entity.YtUser;
import ru.yandex.calendar.util.email.Emails;
import ru.yandex.commune.mail.MailAddress;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.lang.DefaultToString;
import ru.yandex.misc.lang.StringUtils;

/**
 * Universal (just in order to distinguish from {@link Contact} user contact.
 * Represents both external (MK, ya.ru, mail) and calendar internal contacts.
 */
@Getter
@BenderBindAllFields
public class UnivContact extends DefaultToString {
    private final Email email;
    private final String name;
    private final Option<PassportUid> uid;

    public UnivContact(Email email, String name, Option<PassportUid> uid) {
        this.email = email;
        this.name = name;
        this.uid = uid;
    }

    public UnivContact(YtUser user) {
        this(new Email(user.getInfo().getWorkEmail()),
            UserManager.extractPrettyUserName(user),
            Option.of(PassportUid.cons(user.getUid().getValue())));
    }

    public UnivContact(MailAddress mailAddress) {
        this(mailAddress.getEmail(), mailAddress.getPersonal(), Option.empty());
    }

    public String getNameOrEmail() {
        return StringUtils.notBlankO(getName()).getOrElse(() -> Emails.getUnicoded(getEmail()));
    }

    public Sender asSender() {
        return new Sender(getUid(), Option.empty(), new NameI18n(getName(), Option.empty()), Option.empty(), getEmail());
    }

    public MailAddress getMailAddress() {
        return new MailAddress(getEmail(), StringUtils.isNotBlank(getName()) ? getName() : "");
    }

    public static UnivContact formParticipantInfo(ParticipantInfo participant) {
        return new UnivContact(
                participant.getEmail(),
                participant.getName(),
                participant.getUid());
    }

    public static Option<UnivContact> fromUser(YtUser user) {
        try {
            return Option.of(new UnivContact(user));
        } catch (IllegalArgumentException e) {
            return Option.empty();
        }
    }

    public static UnivContact fromParticipantInfoWithHiddenName(ParticipantInfo info) {
        return new UnivContact(info.getEmail(), info.getHiddenName(), info.getUid());
    }
}
